<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\LeaseTemplate;
use App\Models\LeaseTrainingSample;
use Faker\Factory as Faker;

class VakifLeaseTemplateSeeder extends Seeder
{
    public function run()
    {
        $faker = Faker::create('tr_TR');

        // If you created a custom provider:
        $faker->addProvider(new \App\Fakers\TrTaxNumberProvider($faker));


        // Create the template for this lease format
        $template = LeaseTemplate::create([
            'name' => 'Vakıf Standard Lease',
            'description' => 'Standard lease agreement for Vakıf properties',
            'field_mappings' => [
                // Contract fields
                'tenant_name' => [
                    'patterns' => ['Kiracının Adı Soyadı'],
                    'target_field' => 'contract.tenant_id',
                    'type' => 'text'
                ],
                'property_info' => [
                    'patterns' => ['Kiralanan Şeyin Cinsi'],
                    'target_field' => 'contract.property_id',
                    'type' => 'text'
                ],
                'contract_start' => [
                    'patterns' => ['Kiranın Başlangıcı'],
                    'target_field' => 'contract.contract_start',
                    'type' => 'date'
                ],
                'contract_end' => [
                    'patterns' => ['Kira Süresi.*-\s*([0-9]{4}-[0-9]{2}-[0-9]{2})'],
                    'target_field' => 'contract.contract_end',
                    'type' => 'date'
                ],
                'rent_start' => [
                    'patterns' => ['Kiranın Başlangıcı'],
                    'target_field' => 'contract.rent_start',
                    'type' => 'date'
                ],

                // Rental fields
                'rent_amount' => [
                    'patterns' => ['Kira Bedeli.*Yıllık:\s*([0-9,.]+)\s*TRY\s*\(Net\)'],
                    'target_field' => 'rental.rent_amount',
                    'type' => 'currency'
                ],
                'payment_day' => [
                    'patterns' => ['Kira Ödeme Şekli.*ilk\s*\(([0-9]+)-'],
                    'target_field' => 'rental.payment_day',
                    'type' => 'number'
                ],
                'currency' => [
                    'patterns' => ['TRY'],
                    'target_field' => 'rental.currency',
                    'type' => 'text',
                    'default' => 'TRY'
                ],
                'deposit_amount' => [
                    'patterns' => ['Depozito', 'Teminat'],
                    'target_field' => 'rental.deposit_amount',
                    'type' => 'currency',
                    'default' => 0
                ],
                'kdv' => [
                    'patterns' => ['KDV'],
                    'target_field' => 'rental.kdv',
                    'type' => 'number',
                    'default' => 18
                ],
                'kdv_included' => [
                    'patterns' => ['Brüt'],
                    'target_field' => 'rental.kdv_included',
                    'type' => 'boolean',
                    'default' => true
                ],

                // Rent Increase fields
                'increase_date' => [
                    'patterns' => ['Artış Tarihi', 'Zam Tarihi'],
                    'target_field' => 'rent_increase.increase_date',
                    'type' => 'date'
                ],
                'increase_rate' => [
                    'patterns' => ['Artış Oranı', 'Zam Oranı'],
                    'target_field' => 'rent_increase.rate',
                    'type' => 'number'
                ],

                // Fee fields
                'fee_amount' => [
                    'patterns' => ['Aidat Miktarı'],
                    'target_field' => 'fee.amount',
                    'type' => 'currency'
                ],
                'fee_start_date' => [
                    'patterns' => ['Aidat Başlangıç'],
                    'target_field' => 'fee.start_date',
                    'type' => 'date'
                ],

                // Guarantor fields
                'guarantor_name' => [
                    'patterns' => ['MÜŞTEREK BORÇLU MÜTESELSİL KEFİL.*ADI SOYADI', 'Kiracı:'],
                    'target_field' => 'kefiller.0.name',
                    'type' => 'text'
                ],
                'guarantor_phone' => [
                    'patterns' => ['ADRES - TELEFON'],
                    'target_field' => 'kefiller.0.phone',
                    'type' => 'text'
                ],
                'guarantor_id' => [
                    'patterns' => ['TC KİMLİK NO'],
                    'target_field' => 'kefiller.0.id_number',
                    'type' => 'text'
                ],
                'guarantor_address' => [
                    'patterns' => ['ADRES'],
                    'target_field' => 'kefiller.0.address',
                    'type' => 'text'
                ]
            ],
            'sample_patterns' => [
                'EL-HAC NASRULLAHZADE MEHMED BİN NASRULLAH VAKFI',
                'KİRA SÖZLEŞMESİ',
                'KİRAYA Verenin Adı Soyadı',
                'Kiracının Adı Soyadı',
                'Kira Bedeli',
                'Kira Süresi',
                'HUSUSİ ŞARTLAR'
            ]
        ]);

        // Generate 5 training samples with realistic data
        for ($i = 1; $i <= 5; $i++) {
            $buildingNumber = $faker->numberBetween(1, 50);
            $unitLetter = $faker->randomElement(['A', 'B', 'C', 'D']);
            $unitType = $faker->randomElement(['Dükkan', 'Daire', 'Ofis', 'Depo']);
            $floor = $faker->randomElement(['Zemin Kat', '1. Kat', '2. Kat', '3. Kat']);
            $block = $faker->numberBetween(1, 8);
            $netRent = $faker->randomFloat(2, 10000, 50000);
            $grossRent = $netRent * 1.18;
            $startDate = $faker->dateTimeBetween('-1 year', '+1 year')->format('Y-m-d');
            $endDate = $faker->dateTimeBetween($startDate, '+2 years')->format('Y-m-d');
            $paymentDayRange = $faker->randomElement(['01-10', '05-15', '10-20']);

            $sampleText = <<<TEXT
EL-HAC NASRULLAHZADE MEHMED BİN NASRULLAH VAKFI
KİRA SÖZLEŞMESİ

|Mal Sahibi|Mülhak El-Hac Nasrullahzade Mehmed Bin Nasrullah Vakfı|
|---|---|
|Bina Numarası|$buildingNumber|
|Bina İsmi|Nasrullahzade Vakfı Siteleri $block. Blok ($unitLetter)|
|T.C. Kimlik No / Vergi Kimlik No|{$faker->tcNo()}|
|Kiralanan Şeyin Cinsi|$floor $buildingNumber/$unitLetter Nolu $unitType|
|KİRAYA Verenin Adı Soyadı|Mülhak El-Hac Nasrullahzade Mehmed Bin Nasrullah Vakfı adına {$faker->name()}|
|Kiracının Adı Soyadı|{$faker->name()}|
|Kira Bedeli|Yıllık: {$this->formatTurkishNumber($netRent)} TRY (Net) 1 Yıllık: {$this->formatTurkishNumber($grossRent)} TRY (Net) Yıllık: {$this->formatTurkishNumber($netRent)} TRY (Brüt) 1 Yıllık: {$this->formatTurkishNumber($grossRent)} TRY (Brüt)|
|Kira Ödeme Şekli|Her ayın ilk ($paymentDayRange. günü arası) aylık peşin olarak|
|Kira Süresi|$startDate - $endDate (1 Yıllık)|
|Kiranın Başlangıcı|$startDate|
|Kiralanan Şeyin Durumu|$unitType|
|Kullanım Amacı|{$faker->randomElement(['Organizasyon', 'İkamet', 'Ticaret', 'Ofis'])}|

**Kiracı: {$faker->name()}**

===== Page 2 =====

# HUSUSİ ŞARTLAR {$faker->randomNumber(3)}

|MÜŞTEREK BORÇLU MÜTESELSİL KEFİL|KİRACI|KİRAYA VEREN|
|---|---|---|
|TC KİMLİK NO ADI SOYADI ADRES - TELEFON|{$faker->name()} {$faker->phoneNumber()} {$faker->address()}|
TEXT;

            LeaseTrainingSample::create([
                'template_id' => $template->id,
                'sample_text' => $sampleText,
                'verified_data' => [
                    // Contract data
                    'tenant_name' => $faker->name(),
                    'property_info' => "$floor $buildingNumber/$unitLetter Nolu $unitType",
                    'contract_start' => $startDate,
                    'contract_end' => $endDate,
                    'rent_start' => $startDate,

                    // Rental data
                    'rent_amount' => number_format($netRent, 2, '.', ''),
                    'payment_day' => (int)explode('-', $paymentDayRange)[0],
                    'currency' => 'TRY',
                    'deposit_amount' => number_format($netRent, 2, '.', ''),
                    'kdv' => 18,
                    'kdv_included' => true,

                    // Guarantor data
                    'guarantor_name' => $faker->name(),
                    'guarantor_phone' => $faker->phoneNumber(),
                    'guarantor_id' => $faker->taxNumber(),
                    'guarantor_address' => $faker->address()
                ]
            ]);
        }
    }
    private function formatTurkishNumber(float $number): string
    {
        return number_format($number, 2, ',', '.');
    }
}
