<?php

namespace App\Services;

use Abivia\Ledger\Http\Controllers\ReportController;
use Abivia\Ledger\Messages\Report;
use Abivia\Ledger\Models\LedgerDomain;
use Abivia\Ledger\Models\LedgerCurrency;
use App\Models\CustomLedgerAccount;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class TrialBalanceService
{
  /**
   * Generate trial balance using Abivia's native API
   */
  public function generateTrialBalance(
    string $currency,
    int $depth = 3,
    string $fromDate = null,
    string $toDate = null,
    string $domainUuid = null,
    bool $includeZeroBalances = false
  ): array {
    try {
      // Get domain info
      $domain = $domainUuid
        ? LedgerDomain::where('domainUuid', $domainUuid)->first()
        : LedgerDomain::first();

      if (!$domain) {
        throw new \Exception('No ledger domain found');
      }

      // Get currency info
      $currencyModel = LedgerCurrency::where('code', $currency)->first();
      if (!$currencyModel) {
        throw new \Exception("Currency {$currency} not found");
      }

      // Prepare the API request
      $requestData = [
        'name' => 'trialBalance',
        'currency' => $currency,
        'domain' => $domain->code,
        'toDate' => $toDate ?: Carbon::now()->format('Y-m-d'),
        'options' => [
          'depth' => $depth,
          'language' => ['ar', 'en'],
          'decimal' => '.',
          'thousands' => ',',
          'negative' => '-'
        ]
      ];
      $report = \Abivia\Ledger\Messages\Report::fromArray($requestData);
      $controller = new ReportController();

      $response = $controller->generate($report);

      // Make the API request to Abivia Ledger
      //    $response = $this->makeLedgerApiRequest($requestData);
      // dd($response);
      // if (!$response || !isset($response['report']['accounts'])) {
      //   throw new \Exception('Invalid response from ledger API');
      // }

      // Transform the response to our format
      $transformed = $this->transformApiResponse($response, $includeZeroBalances);

      // Ensure we return a valid array
      if (!is_array($transformed)) {
        Log::error('Transform API response returned non-array: ' . gettype($transformed));
        return [];
      }

      return $transformed;
    } catch (\Exception $e) {
      Log::error($e);
      Log::error('Trial Balance API Error: ' . $e->getMessage());
      throw $e;
    }
  }

  /**
   * Generate trial balance with tolerance for floating-point precision issues
   */
  public function generateTrialBalanceWithTolerance(
    string $currency,
    int $depth = 3,
    string $fromDate = null,
    string $toDate = null,
    string $domainUuid = null,
    bool $includeZeroBalances = false,
    float $tolerance = 0.01
  ): array {
    try {
      // First try the normal method
      $result = $this->generateTrialBalance($currency, $depth, $fromDate, $toDate, $domainUuid, $includeZeroBalances);

      // Check if the trial balance is balanced within tolerance
      $totalBalance = collect($result)->sum('total_balance');
      $difference = abs($totalBalance);

      if ($difference <= $tolerance) {
        Log::info("Trial balance generated successfully with tolerance. Difference: {$difference}");
        return $result;
      }

      // If not balanced, try to get balances directly from database
      Log::warning("Trial balance not balanced. Difference: {$difference}. Trying direct database query.");
      return $this->getBalancesDirectly($currency, $domainUuid, $includeZeroBalances, $tolerance);
    } catch (\Exception $e) {
      Log::error('Trial Balance with tolerance failed: ' . $e->getMessage());
      // Fallback to direct database query
      return $this->getBalancesDirectly($currency, $domainUuid, $includeZeroBalances, $tolerance);
    }
  }

  /**
   * Make API request to Abivia Ledger
   */
  private function makeLedgerApiRequest(array $requestData): ?array
  {
    try {
      // Get the ledger API endpoint from config
      $apiPrefix = config('ledger.prefix', 'api/ledger');
      $baseUrl = config('app.url') . '/' . $apiPrefix . '/report';

      // Make the HTTP request
      $response = Http::post($baseUrl, $requestData);

      if ($response->successful()) {
        return $response->json();
      }

      Log::error('Ledger API request failed: ' . $response->body());
      return null;
    } catch (\Exception $e) {
      Log::error('Ledger API request error: ' . $e->getMessage());
      return null;
    }
  }

  /**
   * Transform Abivia API response to our format
   */
  private function transformApiResponse($response, bool $includeZeroBalances): array
  {

    $accounts = $response['accounts'] ?? [];

    $transformed = [];

    foreach ($accounts as $account) {


      // Skip zero balance accounts if not requested
      if (
        !$includeZeroBalances &&
        (float)($account->balance ?? 0) == 0 &&
        (float)($account->total ?? 0) == 0
      ) {
        continue;
      }

      $accountType = $this->getAccountType($account->code ?? '');
      $balance = (float)($account->balance ?? 0);
      $total = (float)($account->total ?? 0);

      // For category accounts, use the total (rolled-up balance) instead of individual balance
      $displayBalance = ($account->category ?? false) ? $total : $balance;

      // Calculate debit and credit balances
      $debitBalance = $displayBalance > 0 ? $displayBalance : 0;
      $creditBalance = $displayBalance < 0 ? abs($displayBalance) : 0;

      $transformed[] = [
        'account_code' => $account->code ?? '',
        'account_name' => $account->name ?? '',
        'account_type' => $accountType,
        'depth' => $account->depth ?? 1,
        'debit_balance' => $debitBalance,
        'credit_balance' => $creditBalance,
        'total_balance' => $displayBalance,
        'opening_balance' => $displayBalance,
        'is_category' => $account->category ?? false,
        'ledger_uuid' => $account->ledgerUuid ?? '',
        'parent_code' => $account->parent ?? '',
        'movements' => [
          'transaction_count' => 0, // Not provided by API
        ]
      ];
    }

    return $transformed;
  }

  /**
   * Get balances directly from database to avoid API validation issues
   */
  private function getBalancesDirectly(
    string $currency,
    string $domainUuid = null,
    bool $includeZeroBalances = false,
    float $tolerance = 0.01
  ): array {
    try {
      // Get domain info
      $domain = $domainUuid
        ? LedgerDomain::where('domainUuid', $domainUuid)->first()
        : LedgerDomain::first();

      if (!$domain) {
        throw new \Exception('No ledger domain found');
      }

      // Get balances directly from ledger_balances table
      $balances = DB::table('ledger_balances')
        ->join('ledger_accounts', 'ledger_balances.ledgerUuid', '=', 'ledger_accounts.ledgerUuid')
        ->where('ledger_balances.currency', $currency)
        ->where('ledger_balances.domainUuid', $domain->domainUuid)
        ->select([
          'ledger_accounts.code',
          'ledger_accounts.extra',
          'ledger_balances.balance',
          'ledger_accounts.debit',
          'ledger_accounts.credit',
          'ledger_accounts.category'
        ])
        ->get();

      $transformed = [];

      foreach ($balances as $balance) {
        $balanceAmount = (float) $balance->balance;

        // Skip zero balance accounts if not requested
        if (!$includeZeroBalances && $balanceAmount == 0) {
          continue;
        }

        $accountType = $this->getAccountType($balance->code);

        // Calculate debit and credit balances
        $debitBalance = $balanceAmount > 0 ? $balanceAmount : 0;
        $creditBalance = $balanceAmount < 0 ? abs($balanceAmount) : 0;

        $transformed[] = [
          'account_code' => $balance->code ?? '',
          'account_name' => $this->getAccountName($balance->code ?? ''),
          'account_type' => $accountType,
          'depth' => 1,
          'debit_balance' => $debitBalance,
          'credit_balance' => $creditBalance,
          'total_balance' => $balanceAmount,
          'opening_balance' => $balanceAmount,
          'is_category' => (bool) $balance->category,
          'ledger_uuid' => $balance->ledgerUuid ?? '',
          'parent_code' => '',
          'movements' => [
            'transaction_count' => 0,
          ]
        ];
      }



      return $transformed;
    } catch (\Exception $e) {
      Log::error('Direct balance query failed: ' . $e->getMessage());
      throw $e;
    }
  }

  /**
   * Get account name from code (you can enhance this)
   */
  private function getAccountName(string $code): string
  {
    // This is a simple mapping - you can enhance this based on your needs
    $names = [
      '1101' => 'Agent 1 Account',
      '1420' => 'TRY Currency Account',
      '4300' => 'USD Currency Account',
      // Add more mappings as needed
    ];

    return $names[$code] ?? "Account {$code}";
  }

  /**
   * Debug method to identify trial balance discrepancies
   */
  public function debugTrialBalance(string $currency, string $domainUuid = null): array
  {
    try {
      $domain = $domainUuid
        ? LedgerDomain::where('domainUuid', $domainUuid)->first()
        : LedgerDomain::first();

      if (!$domain) {
        throw new \Exception('No ledger domain found');
      }

      // Get all balances for the currency
      $balances = DB::table('ledger_balances')
        ->join('ledger_accounts', 'ledger_balances.ledgerUuid', '=', 'ledger_accounts.ledgerUuid')
        ->where('ledger_balances.currency', $currency)
        ->where('ledger_balances.domainUuid', $domain->domainUuid)
        ->select([
          'ledger_accounts.code',
          'ledger_accounts.extra',
          'ledger_balances.balance',
          'ledger_accounts.debit',
          'ledger_accounts.credit',
          'ledger_accounts.category'
        ])
        ->get();

      $analysis = [
        'currency' => $currency,
        'domain' => $domain->code,
        'total_balance' => 0,
        'accounts' => [],
        'is_balanced' => false
      ];

      foreach ($balances as $balance) {
        $balanceAmount = (float) $balance->balance;
        $analysis['total_balance'] += $balanceAmount;

        $accountType = $this->getAccountType($balance->code ?? '');

        $analysis['accounts'][] = [
          'code' => $balance->code,
          'type' => $accountType,
          'balance' => $balanceAmount,
          'is_category' => (bool) $balance->category
        ];
      }

      $analysis['is_balanced'] = abs($analysis['total_balance']) <= 0.01;

      return $analysis;
    } catch (\Exception $e) {
      Log::error('Debug trial balance failed: ' . $e->getMessage());
      throw $e;
    }
  }

  /**
   * Get account type based on account code
   */
  private function getAccountType(string $code): string
  {
    $account = CustomLedgerAccount::where('code', $code)->first();
    return $account ? $account->type : 'Unknown';
  }

  /**
   * Export trial balance to Excel
   */
  public function exportToExcel(array $data, string $filename): \Symfony\Component\HttpFoundation\BinaryFileResponse
  {
    $export = new \App\Exports\TrialBalanceExport($data);
    return \Maatwebsite\Excel\Facades\Excel::download($export, $filename);
  }

  /**
   * Get available currencies
   */
  public function getCurrencies(): \Illuminate\Database\Eloquent\Collection
  {
    return LedgerCurrency::orderBy('code')->get();
  }

  /**
   * Get available domains
   */
  public function getDomains(): \Illuminate\Database\Eloquent\Collection
  {
    return LedgerDomain::orderBy('name')->get();
  }
}
