<?php

namespace App\Services\Transfer;

use Illuminate\Support\Str;
use App\Models\LedgerTransfer;
use Abivia\Ledger\Models\LedgerAccount;
use Abivia\Ledger\Http\Controllers\JournalEntryController;
use Abivia\Ledger\Messages\Entry;
use Abivia\Ledger\Exceptions\Breaker;
use Abivia\Ledger\Http\Controllers\JournalReferenceController;
use Abivia\Ledger\Messages\EntityRef;
use Abivia\Ledger\Messages\Reference;
use Abivia\Ledger\Models\JournalEntry;
use App\Exceptions\InvalidExchangeRateException;
use App\Models\Agent;
use App\Models\Currency;
use App\Models\Exchange;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;

class ExchangeService
{
  public function create(array $data): LedgerTransfer
  {


    $data['secret'] = rand(1000, 9999);

    return LedgerTransfer::create($data);
  }

  public function InsertExchangeEntries($exchange)
  {
    try {
      $controller = new JournalEntryController();
      $timestamp = now()->format('Y-m-d H:i');

      // Create base reference
      $baseReference = $this->createBaseReference($exchange);
      $cash = $this->BuildCashDetails($exchange, $baseReference);
      $details = $this->BuildExchangeLedgerDetails($exchange->currency, $exchange->delivery_currency, $exchange->amount, $exchange->delivery_amount, $exchange);

      $details = array_merge(
        $cash,
        $details
      );



      $groupedByCurrency = collect($details)->groupBy('currency');
      $groupedByCurrency = $groupedByCurrency->toArray();

      foreach ($groupedByCurrency as $currency => $details) {
        $entry_array = Entry::fromArray([
          'currency' => $currency, // 'TRY'
          'entryType' => 'exchange',
          'description' => "Exchange OUT {$exchange->reference}",
          'transDate' => $timestamp,
          'clearing' => true,
          'language' => App::getLocale(),
          'extra' => json_encode(['type' => 'exchange_out', 'exchange_id' => $exchange->id]),
          'details' =>  $details,
        ]);
        $result = $controller->add($entry_array);
        $result->transfer_id = $exchange->id;
        $result->save();
      }
      return true;
    } catch (Breaker $e) {
      Log::info($e);
      //  throw $e;
      //  return  false;
      dd([
        'exception_code' => $e->getCode(),
        'breaker_errors' => $e->getErrors(true), // Force all hidden errors
        'validation_rules' => config('ledger.rules'), // Check config
      ]);
    } catch (\Exception $e) {
      Log::info($e);
      throw $e;
    }
  }
  public  function BuildCashDetails($exchange, $baseReference)
  {
    $from_cash = $this->getAgentAccountCode($exchange->sender_id, '11');
    $to_cash = $this->getAgentAccountCode($exchange->receiver_id, '11');

    return [

      [
        'code' => $from_cash,
        'currency' => $exchange->currency,
        'debit' => $exchange->amount,
        'extra' => json_encode(['detail_type' => 'EXCHANGE_OUT', 'exchange_id' => $exchange->id]),
        'references' => [$this->createDetailReference($baseReference, 'EXCHANGE_OUT')],
      ],
      [
        'code' => $to_cash,
        'currency' => $exchange->delivery_currency,
        'credit' => $exchange->delivery_amount,
        'extra' => json_encode(['detail_type' => 'EXCHANGE_IN', 'exchange_id' => $exchange->id]),
        'references' => [$this->createDetailReference($baseReference, 'EXCHANGE_IN')],
      ],



    ];
  }


  public function CreateBuyingExchangeLedgerEntries(Exchange $exchange): bool
  {

    try {


      $controller = new JournalEntryController();
      $timestamp = now()->format('Y-m-d H:i');

      // Create base reference
      $baseReference = $this->createBaseReference($exchange);


      $booking_debug = ($this->debugBalance('Booking', $this->buildExchangeDetails($exchange, $baseReference)));

      $entryOut = Entry::fromArray([
        'currency' => $exchange->from_currency, // 'TRY'
        'entryType' => 'exchange',
        'description' => "Exchange OUT {$exchange->reference}",
        'transDate' => $timestamp,
        'language' => App::getLocale(),
        'extra' => json_encode(['type' => 'exchange_out', 'exchange_id' => $exchange->id]),
        'details' =>  $this->buildExchangeDetails($exchange, $baseReference)['out'],
      ]);
      $sendResponse = $controller->add($entryOut);
      $entryIn = Entry::fromArray([
        'currency' => $exchange->to_currency, // 'USD'
        'entryType' => 'exchange',
        'description' => "Exchange IN {$exchange->reference}",
        'transDate' => $timestamp,
        'language' => App::getLocale(),
        'extra' => json_encode(['type' => 'exchange_in', 'exchange_id' => $exchange->id]),
        'details' =>  $this->buildExchangeDetails($exchange, $baseReference)['in'],
      ]);

      $sendResponse = $controller->add($entryIn);
      // $senderEntry = Entry::fromArray([
      //     'currency' => $exchange->from_currency,
      //     'entryType' => 'exchange',
      //     'description' => "Exchange {$exchange->reference}",
      //     'transDate' => $timestamp,
      //     'language' => App::getLocale(),
      //     'extra' => json_encode(['type' => 'agent_transfer', 'transfer_id' => $exchange->id]),
      //     'details' => $this->buildExchangeDetails($exchange, $baseReference)
      // ]);
      $status = "completed";






      // $exchange->update([
      //     'debit_journal_uuid' => $sendResponse->journalEntryId,
      //     'status' => $status,
      //     'processed_at' => $timestamp
      // ]);


      return true;
    } catch (Breaker $e) {
      //  return  false;
      Log::error([
        'exception_code' => $e->getCode(),
        'breaker_errors' => $e->getErrors(true), // Force all hidden errors
        'validation_rules' => config('ledger.rules'), // Check config
      ]);
    } catch (\Exception $e) {
      Log::error("Transfer failed: " . $e);
      throw $e;
      return  false;
      // throw $e;
    }
  }

  public  function CreateSellingExchangeLedgerEntries(Exchange $exchange)
  {

    try {

      $controller = new JournalEntryController();
      $timestamp = now()->format('Y-m-d H:i');

      // Create base reference
      $baseReference = $this->createBaseReference($exchange);


      $booking_debug = ($this->debugBalance('Booking', $this->buildSellingExchangeDetails($exchange, $baseReference)));
      Log::info($booking_debug);
      $entryOut = Entry::fromArray([
        'currency' => $exchange->from_currency, // 'TRY'
        'entryType' => 'exchange',
        'description' => "Exchange OUT {$exchange->reference}",
        'transDate' => $timestamp,
        'language' => App::getLocale(),
        'extra' => json_encode(['type' => 'exchange_out', 'exchange_id' => $exchange->id]),
        'details' =>  $this->buildSellingExchangeDetails($exchange, $baseReference)['out'],
      ]);
      $sendResponse = $controller->add($entryOut);
      $result = $sendResponse;
      $result->transfer_id = $exchange->id;
      $result->save();
      $entryIn = Entry::fromArray([
        'currency' => $exchange->to_currency, // 'USD'
        'entryType' => 'exchange',
        'description' => "Exchange IN {$exchange->reference}",
        'transDate' => $timestamp,
        'language' => App::getLocale(),
        'extra' => json_encode(['type' => 'exchange_in', 'exchange_id' => $exchange->id]),
        'details' =>  $this->buildSellingExchangeDetails($exchange, $baseReference)['in'],
      ]);
      $result = $controller->add($entryIn);
      $result->transfer_id = $exchange->id;
      $result->save();
      // $senderEntry = Entry::fromArray([
      //     'currency' => $exchange->from_currency,
      //     'entryType' => 'exchange',
      //     'description' => "Exchange {$exchange->reference}",
      //     'transDate' => $timestamp,
      //     'language' => App::getLocale(),
      //     'extra' => json_encode(['type' => 'agent_transfer', 'transfer_id' => $exchange->id]),
      //     'details' => $this->buildExchangeDetails($exchange, $baseReference)
      // ]);
      $status = "completed";

      // $exchange->update([
      //     'debit_journal_uuid' => $sendResponse->journalEntryId,
      //     'status' => $status,
      //     'processed_at' => $timestamp
      // ]);

      return true;
    } catch (Breaker $e) {
      //  return  false;
      Log::error([
        'exception_code' => $e->getCode(),
        'breaker_errors' => $e->getErrors(true), // Force all hidden errors
        'validation_rules' => config('ledger.rules'), // Check config
      ]);
      return false;
    } catch (\Exception $e) {
      Log::error("Transfer failed: " . $e);
      return  false;
      // throw $e;
    }
  }



  public function CreateCrossExchangeLedgerEntries(Exchange $exchange)
  {
    $base_currency = Currency::where('is_default', 1)->first()->code ?? "USD";

    //from  currency to currency  entry out and entry in
    try {

      $controller = new JournalEntryController();
      $timestamp = now()->format('Y-m-d H:i');

      // Create base reference
      $baseReference = $this->createBaseReference($exchange);

      $this->BuildExchangeLedgerDetails($exchange->from_currency, $exchange->to_currency, $exchange->from_amount, $exchange->to_amount, $exchange);
      $booking_debug = ($this->debugBalance('Booking', $this->buildExchangeDetails($exchange, $baseReference)));
      Log::info($booking_debug);
      $entryOut = Entry::fromArray([
        'currency' => $exchange->from_currency, // 'TRY'
        'entryType' => 'exchange',
        'description' => "Exchange OUT {$exchange->reference}",
        'transDate' => $timestamp,
        'language' => App::getLocale(),
        'extra' => json_encode(['type' => 'exchange_out', 'exchange_id' => $exchange->id]),
        'details' =>  $this->BuildExchangeLedgerDetails($exchange->from_currency, $exchange->to_currency, $exchange->from_amount, $exchange->to_amount, $exchange->id)['out'],
      ]);
      $sendResponse = $controller->add($entryOut);
      $entryIn = Entry::fromArray([
        'currency' => $exchange->to_currency, // 'USD'
        'entryType' => 'exchange',
        'description' => "Exchange IN {$exchange->reference}",
        'transDate' => $timestamp,
        'language' => App::getLocale(),
        'extra' => json_encode(['type' => 'exchange_in', 'exchange_id' => $exchange->id]),
        'details' =>  $this->BuildExchangeLedgerDetails($exchange->from_currency, $exchange->to_currency, $exchange->from_amount, $exchange->to_amount, $exchange->id)['in'],
      ]);
      $sendResponse = $controller->add($entryIn);

      $status = "completed";
    } catch (Breaker $e) {
      //  return  false;
      Log::error([
        'exception_code' => $e->getCode(),
        'breaker_errors' => $e->getErrors(true), // Force all hidden errors
        'validation_rules' => config('ledger.rules'), // Check config
      ]);
      return false;
    } catch (\Exception $e) {
      Log::error("Transfer failed: " . $e);
      throw $e;
      return  false;
      // throw $e;
    }
  }


  public function createBaseReference($transfer): Reference
  {
    try {
      $reference = new Reference();
      $reference->code = $transfer->reference;
      $reference->domain = new EntityRef();
      $reference->domain->code = 'DEFAULT';
      $reference->extra = json_encode([
        'transfer_id' => $transfer->id,
        'type' => 'exchange'
      ]);

      return $reference;
    } catch (Breaker $e) {
      dd([
        'exception_code' => $e->getCode(),
        'breaker_errors' => $e->getErrors(true), // Force all hidden errors
        'validation_rules' => config('ledger.rules'), // Check config
      ]);
    }
  }




  public function BuildExchangeLedgerDetails($from_currency, $to_currency, $from_amount, $to_amount, $exchange)
  {
    try {
      $transfer_id = $exchange->id;
      $base_currency = Currency::where("is_default", 1)->first()->code ?? "USD";
      $currency_accounts = $this->getCurrencyAccountsExchange($from_currency, $to_currency);

      $details = [];
      $detailsOut = [];

      if ($to_currency == $base_currency) {
        $detailsOut = [
          [
            'code' => $currency_accounts['exchange'],
            'currency' => $from_currency,
            'credit' => $from_amount,
            'extra' => json_encode(['detail_type' => 'EXCHANGE_Out', 'transfer_id' => $transfer_id]),
          ],
          [
            'code' => $currency_accounts['exchange'],
            'currency' => $to_currency,
            'debit' => $to_amount,
            'extra' => json_encode(['detail_type' => 'EXCHANGE_Out', 'transfer_id' => $transfer_id]),
          ]
        ];
      } elseif ($from_currency == $base_currency) {
        $cost = getExchangeCostPerUnit($base_currency, $to_currency);
        Log::info($cost);
        if ($cost <= 0) {
          throw new InvalidExchangeRateException($base_currency, $to_currency);
        }
        ValidateCurrencyExchangeBalance($to_currency, $to_amount);
        $base_amount = $to_amount / $cost;
        $gain = $from_amount - $base_amount;
        //dd($gain, $base_amount, $to_amount, $from_amount);
        if ($gain > 0) {
          $gain_code = $currency_accounts['exchange_gain'];
          $gain_type = "credit";
        } else {
          $gain_code = $currency_accounts['exchange_loss'];
          $gain_type = "debit";
        }

        $detailsOut = [
          [
            'code' => $currency_accounts['exchange'],
            'currency' => $to_currency,
            'debit' => $to_amount,
            'extra' => json_encode(['detail_type' => 'EXCHANGE_Out', 'transfer_id' => $transfer_id]),
          ],
          [
            'currency' => $from_currency,
            'code' => $currency_accounts['exchange'],
            'credit' => $base_amount,
            'extra' => json_encode(['detail_type' => 'EXCHANGE_Out', 'transfer_id' => $transfer_id]),
          ],
          [
            'currency' => $from_currency,
            'code' => $gain_code,
            $gain_type => (abs($gain)),
            'extra' => json_encode(['detail_type' => 'EXCHANGE_Out', 'transfer_id' => $transfer_id]),
          ]
        ];
      } else {
        $cost = getExchangeCostPerUnit($base_currency, $from_currency);

        if ($cost <= 0) {
          throw new InvalidExchangeRateException($base_currency, $from_currency);
        }
        $base_amount = $from_amount / $cost;

        // Get results from first recursive call
        $firstDetails = $this->BuildExchangeLedgerDetails($from_currency, $base_currency, $from_amount, $base_amount, $exchange);

        // Get results from second recursive call
        $secondDetails = $this->BuildExchangeLedgerDetails($base_currency, $to_currency, $base_amount, $to_amount, $exchange);

        // Merge the results
        $detailsOut = array_merge($firstDetails, $secondDetails);
      }


      return $detailsOut;
    } catch (Breaker $e) {
      Log::error([
        'exception_code' => $e->getCode(),
        'breaker_errors' => $e->getErrors(true),
        'validation_rules' => config('ledger.rules'),
      ]);
    }
  }
  public function BuildExchangeLedgerDetails2($from_currency, $to_currency, $from_amount, $to_amount, $transfer_id)
  {
    try {
      $base_currency = Currency::where("is_default", 1)->first()->code ?? "USD";
      $currency_accounts = $this->getCurrencyAccountsExchange($from_currency, $to_currency);
      $accounts = $this->getValidatedAccounts($transfer_id);
      $details = [];
      $detailsOut = [];

      if ($to_currency == $base_currency) {
        $detailsOut = [
          [
            'code' => $accounts['agent']['from_cash'],
            'currency' => $from_currency,
            'debit' => $from_amount,
            'extra' => json_encode(['detail_type' => 'EXCHANGE_Out', 'transfer_id' => $transfer_id]),
          ],
          [
            'code' => $currency_accounts['buy'],
            'currency' => $from_currency,
            'credit' => $from_amount,
            'extra' => json_encode(['detail_type' => 'EXCHANGE_Out', 'transfer_id' => $transfer_id]),
          ],
          [
            'code' => $currency_accounts['to_cash'],
            'currency' => $to_currency,
            'credit' => $to_amount,
            'extra' => json_encode(['detail_type' => 'EXCHANGE_In', 'transfer_id' => $transfer_id]),
          ],
          [
            'code' => $currency_accounts['sell'],
            'currency' => $to_currency,
            'debit' => $to_amount,
            'extra' => json_encode(['detail_type' => 'EXCHANGE_In', 'transfer_id' => $transfer_id]),
          ]

        ];
      } elseif ($from_currency == $base_currency) {
        $cost = getExchangeCostPerUnit($base_currency, $to_currency);
        if ($cost == 0) {
          throw new InvalidExchangeRateException($base_currency, $to_currency);
        }
        $base_amount = $to_amount / $cost;
        $gain = $from_amount - $base_amount;

        if ($gain > 0) {
          $gain_code = $currency_accounts['exchange_gain'];
          $gain_type = "credit";
        } else {
          $gain_code = $currency_accounts['exchange_loss'];
          $gain_type = "debit";
        }

        $detailsOut = [
          [
            'code' => $currency_accounts['sell'],
            'currency' => $to_currency,
            'debit' => $to_amount,
            'extra' => json_encode(['detail_type' => 'EXCHANGE_Out', 'transfer_id' => $transfer_id]),
          ],
          [
            'currency' => $from_currency,
            'code' => $currency_accounts['sell'],
            'credit' => $base_amount,
            'extra' => json_encode(['detail_type' => 'EXCHANGE_Out', 'transfer_id' => $transfer_id]),
          ],
          [
            'currency' => $from_currency,
            'code' => $gain_code,
            $gain_type => (abs($gain)),
            'extra' => json_encode(['detail_type' => 'EXCHANGE_Out', 'transfer_id' => $transfer_id]),
          ]
        ];
      } else {
        $cost = getExchangeCostPerUnit($base_currency, $from_currency);
        if ($cost == 0) {
          throw new InvalidExchangeRateException($base_currency, $from_currency);
        }
        $base_amount = $from_amount / $cost;

        // Get results from first recursive call
        $firstDetails = $this->BuildExchangeLedgerDetails($from_currency, $base_currency, $from_amount, $base_amount, $transfer_id);

        // Get results from second recursive call
        $secondDetails = $this->BuildExchangeLedgerDetails($base_currency, $to_currency, $base_amount, $to_amount, $transfer_id);

        // Merge the results
        $detailsOut = array_merge($firstDetails['out'], $secondDetails['out']);
      }

      $detailsOut;

      return $detailsOut;
    } catch (Breaker $e) {
      dd([
        'exception_code' => $e->getCode(),
        'breaker_errors' => $e->getErrors(true),
        'validation_rules' => config('ledger.rules'),
      ]);
    } catch (\Exception $e) {
      Log::error("Transfer failed: " . $e);
      return [];
    }
  }

  public  function getCurrencyAccountsExchange($from_currency, $to_currency): array
  {
    $from_currency = Currency::where("code", $from_currency)->first();
    $to_currency = Currency::where("code", $to_currency)->first();
    $base_currency = Currency::where("is_default", 1)->first()->code ?? "USD";
    $currency = $from_currency;
    if ($from_currency->code == $base_currency) {
      $currency = $to_currency;
    }
    return [
      'sell' => (string)$currency->GetExchangeAccountCode()['exchange'],
      'buy' => (string)$currency->GetExchangeAccountCode()['exchange'],
      'exchange' => (string)$currency->GetExchangeAccountCode()['exchange'],
      'exchange_gain' => "4110",
      "exchange_loss" => "4120",
    ];
  }
  public function buildExchangeDetails(Exchange $exchange, Reference $baseReference): array
  {
    try {
      $accounts = $this->getValidatedAccounts($exchange);
      $currency_accounts = $this->getCurrencyAccounts($exchange);

      $details = [];
      $from_amount = $exchange->from_amount;
      $to_amount = $exchange->to_amount;

      $detailsOut = [
        [
          'code' => (string)$accounts['agent']['from_cash'], // TRY cash
          'debit' => $exchange->from_amount,
          'extra' => json_encode(['detail_type' => 'EXCHANGE_OUT', 'exchange_id' => $exchange->id]),
          'references' => [$this->createDetailReference($baseReference, 'EXCHANGE_TRANSFER')],
        ],
        [
          'code' => $currency_accounts['buy'], // TRY buy
          'credit' => $exchange->from_amount,
          'extra' => json_encode(['detail_type' => 'EXCHANGE_OUT', 'exchange_id' => $exchange->id]),
          'references' => [$this->createDetailReference($baseReference, 'EXCHANGE_TRANSFER')],
        ],
      ];
      $detailsIn = [
        [
          'code' => $currency_accounts['buy'], // USD buy
          'debit' => $exchange->to_amount,
          'extra' => json_encode(['detail_type' => 'EXCHANGE_IN', 'exchange_id' => $exchange->id]),
          'references' => [$this->createDetailReference($baseReference, 'EXCHANGE_TRANSFER')],

        ],
        [
          'code' => (string)$accounts['agent']['to_cash'], // USD cash
          'credit' => $exchange->to_amount,
          'extra' => json_encode(['detail_type' => 'EXCHANGE_IN', 'exchange_id' => $exchange->id]),
          'references' => [$this->createDetailReference($baseReference, 'EXCHANGE_TRANSFER')],

        ],
      ];
      $details['out'] = $detailsOut;
      $details['in'] = $detailsIn;


      return $details;
    } catch (Breaker $e) {
      //  return  false;
      dd([
        'exception_code' => $e->getCode(),
        'breaker_errors' => $e->getErrors(true), // Force all hidden errors
        'validation_rules' => config('ledger.rules'), // Check config
      ]);
    } catch (\Exception $e) {
      Log::error("Transfer failed: " . $e);
      throw $e;
      return [];
      // throw $e;
    }
  }

  public function buildSellingExchangeDetails(Exchange $exchange, Reference $baseReference)
  {
    try {
      $accounts = $this->getValidatedAccounts($exchange);

      $currency_accounts = $this->getCurrencyAccounts($exchange);

      $cost = getExchangeCostPerUnit($exchange->from_currency, $exchange->to_currency);

      $details = [];
      $net_from_amount = ($exchange->to_amount / $cost);

      $gain =  $exchange->from_amount - $net_from_amount;

      if ($gain > 0) {
        $gain_account = $currency_accounts['exchange_gain'];
      } else {
        $gain_account = $currency_accounts['exchange_loss'];
      }

      // Selling USD for TRY:
      // 1) Credit USD cash (agent's to_cash) — USD outflow
      // 2) Credit USD buy account — USD asset decrease
      $detailsOut = [
        [
          'code' => (string)$accounts['agent']['from_cash'], // USD cash (agent)
          'debit' => $exchange->from_amount,  // USD amount sold
          'extra' => json_encode(['detail_type' => 'EXCHANGE_OUT', 'exchange_id' => $exchange->id]),
          'references' => [$this->createDetailReference($baseReference, 'EXCHANGE_TRANSFER')],
        ],

        [
          'code' => $currency_accounts['sell'], // USD buy account (reduce USD asset)
          'credit' => $net_from_amount, // USD amount sold minus cost
          'extra' => json_encode(['detail_type' => 'EXCHANGE_OUT', 'exchange_id' => $exchange->id]),
          'references' => [$this->createDetailReference($baseReference, 'EXCHANGE_TRANSFER')],
        ],
        [
          'code' => $gain_account, // Exchange account (USD asset decrease)
          'credit' => $gain, // USD cost
          'extra' => json_encode(['detail_type' => 'EXCHANGE_OUT', 'exchange_id' => $exchange->id]),
          'references' => [$this->createDetailReference($baseReference, 'EXCHANGE_TRANSFER')],
        ],
      ];

      // 3) Debit TRY buy account — increase TRY asset (cash in)
      // 4) Debit TRY cash (agent from_cash) — increase TRY cash
      $detailsIn = [
        [
          'code' => $currency_accounts['sell'], // TRY sell account
          'debit' => $exchange->to_amount, // TRY amount received
          'extra' => json_encode(['detail_type' => 'EXCHANGE_IN', 'exchange_id' => $exchange->id]),
          'references' => [$this->createDetailReference($baseReference, 'EXCHANGE_TRANSFER')],
        ],
        [
          'code' => (string)$accounts['agent']['to_cash'], // TRY cash (agent)
          'credit' => $exchange->to_amount,
          'extra' => json_encode(['detail_type' => 'EXCHANGE_IN', 'exchange_id' => $exchange->id]),
          'references' => [$this->createDetailReference($baseReference, 'EXCHANGE_TRANSFER')],
        ],
      ];

      $details['out'] = $detailsOut;
      $details['in'] = $detailsIn;

      return $details;
    } catch (Breaker $e) {
      Log::error([
        'exception_code' => $e->getCode(),
        'breaker_errors' => $e->getErrors(true),
        'validation_rules' => config('ledger.rules'),
      ]);
    } catch (\Exception $e) {
      Log::error("Selling exchange failed: " . $e);
      throw $e;
      return [];
    }
  }








  public function debugBalance(string $title, array $details)
  {
    // Flatten if 'in' and 'out' keys exist
    if (isset($details['in']) && isset($details['out'])) {
      $details = array_merge($details['out'], $details['in']);
    }

    $sum = 0;
    $text = "$title Details:\n";

    foreach ($details as $d) {
      $amount = $d['debit'] ?? ($d['credit'] ?? 0);
      $type = isset($d['debit']) ? 'Debit' : 'Credit';
      $sum += isset($d['debit']) ? $amount : -$amount;

      $account = LedgerAccount::with("names")->where('code', $d['code'])->first();
      $accountName = $account ? $account->names->first()->name : 'Unknown Account';

      $text .= "- {$d['code']} ({$accountName}): {$type} {$amount}\n";
    }

    $text .= "Total Balance: $sum\n";
    $text .= "------------------\n";

    return $text;
  }



  public function createDetailReference(Reference $baseReference, string $suffix): Reference
  {
    $reference = new Reference();
    $reference->code = $baseReference->code . '-' . $suffix;
    $reference->domain = clone $baseReference->domain;
    $reference->extra = json_encode(array_merge(
      json_decode($baseReference->extra, true) ?? [],
      ['detail_type' => $suffix]
    ));

    //     $controller = new JournalReferenceController();
    //    $reference = $controller->add($reference);


    return $reference;
  }


  public function getValidatedAccounts(Exchange $exchange): array
  {


    return [
      'agent' => [
        'from_cash' => (string) $this->getAgentAccountCode($exchange->agent_id, '11'),
        'to_cash' => (string) $this->getAgentAccountCode($exchange->agent_id, '11'),
      ],

    ];
  }
  public  function getCurrencyAccounts(Exchange $exchange): array
  {
    $from_currency = Currency::where("code", $exchange->from_currency)->first();
    $to_currency = Currency::where("code", $exchange->to_currency)->first();
    return [
      'sell' => (string)$to_currency->GetExchangeAccountCode()['sell'],
      'buy' => (string)$from_currency->GetExchangeAccountCode()['buy'],
      'exchange' => (string)$to_currency->GetExchangeAccountCode()['exchange'],
      'exchange_gain' => "4210",
      "exchange_loss" => "4220",
    ];
  }

  public function getAgentAccountCode(int $agentId, string $prefix): string
  {
    $agent = Agent::findOrFail($agentId);
    $real_agent_id = $agent->rootAgent()->id;
    $account = LedgerAccount::where('extra->agent_id', $real_agent_id)
      ->where('code', 'like', $prefix . '%')
      ->firstOrFail();

    return $account->code;
  }
  public function getCurrencyAccountCode(int $currevncyId, string $prefix): string
  {

    $account = LedgerAccount::where('extra->currency_id', $currevncyId)
      ->where('code', 'like', $prefix . '%')
      ->firstOrFail();

    return $account->code;
  }
  public function generateRandomTransferNumber(): string
  {
    $branchCode = str_pad(mt_rand(0, 999), 3, '0', STR_PAD_LEFT);
    $transactionId = str_pad(mt_rand(0, 999999), 6, '0', STR_PAD_LEFT);

    return "EXG-{$branchCode}-{$transactionId}";
  }
}
