<?php

namespace App\Services;

use Mpdf\Mpdf;
use App\DataTables\TransactionDataTable;
use App\Models\CustomLedgerAccount;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;

class StatementPdfService
{
  protected $transactionDataTable;

  public function __construct(TransactionDataTable $transactionDataTable)
  {
    $this->transactionDataTable = $transactionDataTable;
  }

  public function generateStatementPdf($filters = [])
  {
    try {
      $statementDetails = $this->getStatementDetails($filters);
      $transactions = $this->getTransactions($filters);

      // Set environment variables BEFORE creating mPDF instance
      putenv('MPDF_TEMP_DIR=/tmp/mpdf');
      putenv('MPDF_CACHE_DIR=/tmp/mpdf/cache');

      // Ensure temp directories exist and are writable
      $this->ensureTempDirectoriesExist();

      // Create mPDF instance with RTL and Arabic support
      $mpdf = new Mpdf([
        'mode' => 'utf-8',
        'format' => 'A4',
        'margin_left' => 15,
        'margin_right' => 15,
        'margin_top' => 15,
        'margin_bottom' => 15,
        'margin_header' => 10,
        'margin_footer' => 10,
        'default_font' => 'dejavusans',
        'default_font_size' => 10,
        'temp_dir' => '/tmp/mpdf',           // Main temp directory
        'cache_dir' => '/tmp/mpdf/cache',    // Cache directory
        'chroot' => public_path(),
        'direction' => 'rtl',
        'autoScriptToLang' => true,
        'autoLangToFont' => true,
      ]);

      // Load the HTML view
      $html = view('pdfs.statement', [
        'statementDetails' => $statementDetails,
        'transactions' => $transactions,
        'filters' => $filters
      ])->render();

      // Write HTML to mPDF
      $mpdf->WriteHTML($html);

      return $mpdf;
    } catch (\Exception $e) {
      Log::error('Error generating statement PDF: ' . $e->getMessage(), [
        'trace' => $e->getTraceAsString(),
        'filters' => $filters
      ]);
      throw $e;
    }
  }

  protected function getStatementDetails($filters)
  {
    try {
      if (empty($filters['account']) || empty($filters['currency'])) {
        return $this->getDefaultStatementDetails();
      }

      $accountCode = $filters['account'];
      $currency = $filters['currency'];
      $account = CustomLedgerAccount::where('code', $accountCode)->first();

      if (!$account) {
        return $this->getDefaultStatementDetails();
      }

      $accountName = $account->names->firstWhere('language', app()->getLocale())?->name ?? $accountCode;
      $accountType = $account->extra['type'] ?? 'Unknown';

      $fromDate = null;
      $toDate = null;

      if (isset($filters['dateFilter']) && $filters['dateFilter'] === 'custom') {
        $fromDate = $filters['fromDate'] ? Carbon::parse($filters['fromDate']) : null;
        $toDate = $filters['toDate'] ? Carbon::parse($filters['toDate']) : null;
      }

      // Get comprehensive statement data using the same logic as the details page
      $statementData = $this->getComprehensiveStatementData($filters, $account);

      return [
        'account_code' => $accountCode,
        'account_name' => $accountName,
        'account_type' => $accountType,
        'currency' => $currency,
        'from_date' => $fromDate,
        'to_date' => $toDate,
        'generated_at' => now(),
        'include_unreceived' => $filters['includeUnreceived'] ?? false,
        // Additional comprehensive data
        'opening_balance' => $statementData['opening_balance'] ?? 0,
        'current_balance' => $statementData['current_balance'] ?? 0,
        'balance_type' => $statementData['balance_type'] ?? 'صفر',
        'entry_count' => $statementData['entry_count'] ?? 0,
        'out_transfers' => $statementData['out_transfers'] ?? 0,
        'in_transfers' => $statementData['in_transfers'] ?? 0,
        'total_debit' => $statementData['total_debit'] ?? 0,
        'total_credit' => $statementData['total_credit'] ?? 0,
        'currency_words' => $statementData['currency_words'] ?? '',
      ];
    } catch (\Exception $e) {
      Log::error('Error getting statement details: ' . $e->getMessage());
      return $this->getDefaultStatementDetails();
    }
  }

  protected function getComprehensiveStatementData($filters, $account)
  {
    try {
      // Use the same logic as the Details component
      $fromDate = $filters['fromDate'] ?? Carbon::now()->subMonth()->toDateString();
      $toDate = $filters['toDate'] ?? Carbon::now()->toDateString();

      // Get transactions for the period to calculate totals
      $this->transactionDataTable->filters = $filters;
      $transactions = $this->transactionDataTable->query();

      $totalDebit = 0;
      $totalCredit = 0;
      $entryCount = $transactions->count();
      $outTransfers = 0;
      $inTransfers = 0;

      foreach ($transactions as $transaction) {
        $isArray = is_array($transaction);

        // Handle Abivia Ledger report data structure
        // The data might have different field names depending on the report structure
        $debit = 0;
        $credit = 0;

        if ($isArray) {
          // Try different possible field names for debit/credit
          $debit = $transaction['debit'] ?? $transaction['debit_amount'] ?? $transaction['amount'] ?? 0;
          $credit = $transaction['credit'] ?? $transaction['credit_amount'] ?? 0;

          // If amount is positive, it's a debit; if negative, it's a credit
          if (isset($transaction['amount']) && !isset($transaction['debit']) && !isset($transaction['credit'])) {
            $amount = floatval($transaction['amount']);
            if ($amount > 0) {
              $debit = $amount;
            } else {
              $credit = abs($amount);
            }
          }
        } else {
          // Handle object data
          $debit = $transaction->debit ?? $transaction->debit_amount ?? $transaction->amount ?? 0;
          $credit = $transaction->credit ?? $transaction->credit_amount ?? 0;

          // If amount is positive, it's a debit; if negative, it's a credit
          if (isset($transaction->amount) && !isset($transaction->debit) && !isset($transaction->credit)) {
            $amount = floatval($transaction->amount);
            if ($amount > 0) {
              $debit = $amount;
            } else {
              $credit = abs($amount);
            }
          }
        }

        $totalDebit += floatval($debit);
        $totalCredit += floatval($credit);

        // Count transfers (you might need to adjust this based on your data structure)
        if (floatval($debit) > 0) $outTransfers++;
        if (floatval($credit) > 0) $inTransfers++;
      }

      // Calculate balance difference
      $difference = $totalDebit - $totalCredit;

      if ($difference > 0) {
        $balanceType = 'لنا';
        $currentBalance = $difference;
      } elseif ($difference < 0) {
        $balanceType = 'علينا';
        $currentBalance = abs($difference);
      } else {
        $balanceType = 'صفر';
        $currentBalance = 0;
      }

      // Get currency words
      $currencyWords = $this->getCurrencyWords($filters['currency']);

      return [
        'opening_balance' => 0, // You might need to implement this based on your business logic
        'current_balance' => $currentBalance,
        'balance_type' => $balanceType,
        'entry_count' => $entryCount,
        'out_transfers' => $outTransfers,
        'in_transfers' => $inTransfers,
        'total_debit' => $totalDebit,
        'total_credit' => $totalCredit,
        'currency_words' => $currencyWords,
      ];
    } catch (\Exception $e) {
      Log::error('Error getting comprehensive statement data: ' . $e->getMessage());
      return [
        'opening_balance' => 0,
        'current_balance' => 0,
        'balance_type' => 'صفر',
        'entry_count' => 0,
        'out_transfers' => 0,
        'in_transfers' => 0,
        'total_debit' => 0,
        'total_credit' => 0,
        'currency_words' => '',
      ];
    }
  }

  protected function getTransactions($filters)
  {
    try {
      $this->transactionDataTable->filters = $filters;
      $data = $this->transactionDataTable->query();

      Log::info('PDF Service - Raw transaction data:', [
        'count' => $data->count(),
        'sample' => $data->take(2)->toArray()
      ]);

      $transactions = collect($data)->map(function ($item) {
        // Handle both object and array data structures
        $isArray = is_array($item);

        // Handle Abivia Ledger report data structure
        $debit = 0;
        $credit = 0;
        $balance = 0;

        if ($isArray) {
          // Try different possible field names for debit/credit
          $debit = $item['debit'] ?? $item['debit_amount'] ?? 0;
          $credit = $item['credit'] ?? $item['credit_amount'] ?? 0;
          $balance = $item['balance'] ?? 0;

          // If amount is positive, it's a debit; if negative, it's a credit
          if (isset($item['amount']) && !isset($item['debit']) && !isset($item['credit'])) {
            $amount = floatval($item['amount']);
            if ($amount > 0) {
              $debit = $amount;
            } else {
              $credit = abs($amount);
            }
          }
        } else {
          // Handle object data
          $debit = $item->debit ?? $item->debit_amount ?? 0;
          $credit = $item->credit ?? $item->credit_amount ?? 0;
          $balance = $item->balance ?? 0;

          // If amount is positive, it's a debit; if negative, it's a credit
          if (isset($item->amount) && !isset($item->debit) && !isset($item->credit)) {
            $amount = floatval($item->amount);
            if ($amount > 0) {
              $debit = $amount;
            } else {
              $credit = abs($amount);
            }
          }
        }

        return [
          'id' => $isArray ? ($item['id'] ?? 'N/A') : ($item->id ?? 'N/A'),
          'date' => $isArray ?
            (isset($item['created_at']) ? Carbon::parse($item['created_at'])->format('Y-m-d H:i:s') : (isset($item['date']) ? Carbon::parse($item['date'])->format('Y-m-d H:i:s') : 'N/A')) : ($item->created_at ? Carbon::parse($item->created_at)->format('Y-m-d H:i:s') : ($item->date ? Carbon::parse($item->date)->format('Y-m-d H:i:s') : 'N/A')),
          'reference' => $isArray ? ($item['transfer_number'] ?? $item['reference'] ?? 'N/A') : ($item->transfer_number ?? $item->reference ?? 'N/A'),
          'type' => $isArray ? ($item['type'] ?? 'N/A') : ($item->type ?? 'N/A'),
          'transfer_type' => $isArray ?
            (isset($item['transfer_type']) ? strip_tags($item['transfer_type']) : 'N/A') : (isset($item->transfer_type) ? strip_tags($item->transfer_type) : 'N/A'),
          'direction' => $isArray ? ($item['direction'] ?? 'N/A') : ($item->direction ?? 'N/A'),
          'sender' => $isArray ? ($item['sender'] ?? 'N/A') : ($item->sender ?? 'N/A'),
          'receiver' => $isArray ? ($item['receiver'] ?? 'N/A') : ($item->receiver ?? 'N/A'),
          'description' => $isArray ? ($item['description'] ?? 'N/A') : ($item->description ?? 'N/A'),
          'debit' => $debit,
          'credit' => $credit,
          'balance' => $balance,
          'user' => $isArray ?
            (isset($item['user']['name']) ? $item['user']['name'] : (isset($item['user']) ? $item['user'] : 'N/A')) : ($item->user->name ?? 'N/A'),
          'status' => $isArray ? ($item['status'] ?? 'N/A') : ($item->status ?? 'N/A')
        ];
      });

      Log::info('PDF Service - Transformed transactions:', [
        'count' => $transactions->count()
      ]);

      return $transactions;
    } catch (\Exception $e) {
      Log::error('Error getting transactions: ' . $e->getMessage(), [
        'trace' => $e->getTraceAsString()
      ]);
      return collect();
    }
  }

  public function formatAmount($amount, $currency = 'TRY')
  {
    if ($amount === null || $amount === '') {
      return '0.00';
    }

    $amount = floatval($amount);

    if ($currency === 'USD') {
      return number_format($amount, 2, '.', ',');
    } else {
      return number_format($amount, 2, ',', '.');
    }
  }

  public function formatDate($date)
  {
    if (!$date) {
      return 'N/A';
    }

    try {
      return Carbon::parse($date)->format('Y-m-d H:i:s');
    } catch (\Exception $e) {
      return 'N/A';
    }
  }

  protected function getAccountName($accountCode)
  {
    try {
      $account = CustomLedgerAccount::where('code', $accountCode)->first();
      if ($account) {
        return $account->names->firstWhere('language', app()->getLocale())?->name ?? $accountCode;
      }
      return $accountCode;
    } catch (\Exception $e) {
      return $accountCode;
    }
  }

  protected function getAccountType($accountCode)
  {
    try {
      $account = CustomLedgerAccount::where('code', $accountCode)->first();
      return $account->extra['type'] ?? 'Unknown';
    } catch (\Exception $e) {
      return 'Unknown';
    }
  }

  protected function getCurrencyWords($currency)
  {
    $currencyNames = [
      'USD' => 'دولار أمريكي',
      'TRY' => 'ليرة تركية',
      'EUR' => 'يورو',
      'GBP' => 'جنيه إسترليني'
    ];

    return $currencyNames[$currency] ?? $currency;
  }

  protected function getDefaultStatementDetails()
  {
    return [
      'account_code' => 'N/A',
      'account_name' => 'N/A',
      'account_type' => 'N/A',
      'currency' => 'N/A',
      'from_date' => null,
      'to_date' => null,
      'generated_at' => now(),
      'include_unreceived' => false
    ];
  }

  protected function getTempDirectory()
  {
    // Try Laravel storage first
    $tempDir = storage_path('app/temp/mpdf');

    // If Laravel storage is not writable, use system temp
    if (!is_writable($tempDir)) {
      $tempDir = sys_get_temp_dir() . '/mpdf';
    }

    if (!file_exists($tempDir)) {
      mkdir($tempDir, 0777, true);
    }

    return $tempDir;
  }

  protected function ensureTempDirectoriesExist()
  {
    $tempDir = $this->getTempDirectory();
    if (!is_writable($tempDir)) {
      Log::warning('MPDF temp directory is not writable. Attempting to fix permissions.', [
        'path' => $tempDir
      ]);
      if (!chmod($tempDir, 0777)) {
        Log::error('Failed to set permissions for MPDF temp directory.', [
          'path' => $tempDir,
          'error' => error_get_last()
        ]);
        throw new \Exception('MPDF temp directory is not writable and could not be fixed.');
      }
      Log::info('MPDF temp directory permissions fixed.', [
        'path' => $tempDir
      ]);
    }

    $cacheDir = storage_path('app/temp/mpdf/cache');
    if (!file_exists($cacheDir)) {
      mkdir($cacheDir, 0777, true);
    }
    if (!is_writable($cacheDir)) {
      Log::warning('MPDF cache directory is not writable. Attempting to fix permissions.', [
        'path' => $cacheDir
      ]);
      if (!chmod($cacheDir, 0777)) {
        Log::error('Failed to set permissions for MPDF cache directory.', [
          'path' => $cacheDir,
          'error' => error_get_last()
        ]);
        throw new \Exception('MPDF cache directory is not writable and could not be fixed.');
      }
      Log::info('MPDF cache directory permissions fixed.', [
        'path' => $cacheDir
      ]);
    }
  }
}
