<?php

namespace App\Services\Ledger;

use Abivia\Ledger\Http\Controllers\LedgerAccountController;
use Abivia\Ledger\Messages\Account;
use App\Models\Agent;
use App\Models\Currency;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Abivia\Ledger\Exceptions\Breaker;
use Abivia\Ledger\Models\LedgerAccount;

class LedgerService
{
  public static function createCurrencySubAccountsForAgent(Agent $agent): void
  {
    $categories = [
      ['code' => '1100', 'debit' => true,  'type' => 'cash'],
      ['code' => '1200', 'debit' => true,  'type' => 'receivable'],
      ['code' => '1300', 'debit' => true,  'type' => 'undelivered'],
      //['code' => '1400', 'debit' => true,  'type' => 'exchange'],
      ['code' => '2100', 'debit' => false, 'type' => 'payable'],
      ['code' => '4100', 'debit' => false, 'type' => 'unearned_fee'],

      ['code' => '5100', 'debit' => true,  'type' => 'agent_earned_commission'],
      ['code' => '5200', 'debit' => true,  'type' => 'agent_unearned_commission'],
      ['code' => '5300', 'debit' => true,  'type' => 'exchnage_loss'],
    ];

    foreach ($categories as $category) {
      Log::info("Creating currency accounts for agent {$agent->code} under category {$category['code']} ({$category['type']})");
      self::createCurrencyAccountsUnderAgentCategory($agent, $category['code'], $category['type'], $category['debit']);
    }
  }

  protected static function createCurrencyAccountsUnderAgentCategory(Agent $agent, string $parentCode, string $type, bool $isDebit): void
  {
    $controller = new LedgerAccountController();
    $currencies = Currency::all();
    $agentCategoryCode = self::generateAgentAccountCode($parentCode, $agent->id);
    Log::info("Creating agent category account for agent {$agent->code} with code {$agentCategoryCode}");
    foreach ($currencies as $currency) {
      $accountCode = self::generateAgentCurrencyCode($parentCode, $agent->id, $currency->id);
      Log::info("Creating account for agent {$agent->code} in currency {$currency->code} with code {$accountCode}");
      try {
        if (! LedgerAccount::where('code', $accountCode)->exists()) {
          $controller->add(Account::fromArray([
            'code' => $accountCode,
            'names' => self::getLocalizedAccountNames($agent, $currency, $type),
            'parent' => $agentCategoryCode,
            'debit' => $isDebit,
            'credit' => !$isDebit,
            'category' => false,
            'reference' => $agent->ledger_reference_uuid,
            'extra' => json_encode([
              'agent_id' => $agent->id,
              'currency_code' => $currency->code,
              'currency_id' => $currency->id,
              'account_type' => $type
            ])
          ]));
        } else {
          Log::info("Account {$accountCode} already exists for agent {$agent->code} in currency {$currency->code}");
        }
      } catch (Breaker $e) {
        Log::error([
          'account_code' => $accountCode,
          'parent_code' => $agentCategoryCode,
          'agent_code' => $agent->code,
          'currency_code' => $currency->code,
          'exception_code' => $e->getCode(),
          'breaker_errors' => $e->getErrors(true), // Force all hidden errors
          'validation_rules' => config('ledger.rules'), // Check config
        ]);
      }
    }
  }

  protected static function getLocalizedAccountNames(Agent $agent, Currency $currency, string $type): array
  {
    $typeNames = [
      'cash' => ['en' => 'Cash',        'ar' => 'نقدية'],
      'receivable' => ['en' => 'Receivable',  'ar' => 'مستحقات'],
      'undelivered' => ['en' => 'Undelivered', 'ar' => 'غير مسلمة'],
      'exchange' => ['en' => 'Exchange',    'ar' => 'تبادل عملات'],
      'payable' => ['en' => 'Payable',     'ar' => 'مستحق لـ'],
      'unearned_fee' => ['en' => 'Unearned Fee', 'ar' => 'رسوم غير محققة'],
      'earned_fee' => ['en' => 'Earned Fee',   'ar' => 'رسوم محققة'],
      'agent_earned_commission' => ['en' => 'Agent Earned Commission', 'ar' => 'عمولة وكيل محققة'],
      'agent_unearned_commission' => ['en' => 'Agent Unearned Commission', 'ar' => 'عمولة وكيل غير محققة'],
    ];

    $name = $typeNames[$type] ?? ['en' => 'Sub Account', 'ar' => 'حساب فرعي'];

    return [
      ['language' => 'en', 'name' => "{$agent->code} {$currency->code} {$name['en']}"],
      ['language' => 'ar', 'name' => "{$name['ar']} {$agent->code} {$currency->code}"],
    ];
  }

  public static function generateAgentAccountCode(string $baseCode, int $agentId): string
  {
    return substr($baseCode, 0, 4) . str_pad($agentId, 7, '0', STR_PAD_LEFT);
  }

  public static function generateAgentCurrencyCode(string $baseCode, int $agentId, int $currencyId): string
  {
    return substr($baseCode, 0, 4)
      . str_pad($agentId, 7, '0', STR_PAD_LEFT)
      . str_pad($currencyId, 7, '0', STR_PAD_LEFT);
  }
}
