<?php

namespace App\Observers;

use App\Services\Ledger\AgentLedgerService;
use App\Services\Ledger\LedgerService;
use App\Jobs\LogoutAgentUsersJob;
use App\Helpers\RedisSessionHelper;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Session;

class AgentObserver
{
  public function created($model)
  {
    if (($model->isAgent() || $model->isCashier())   && $model->isActive()) {
      if ($model->isCashier()) {
        $parent = $model->rootAgent();
        $main_cashier_agent = $parent->getMainCashierAgent();
        if (!$main_cashier_agent) {
          $main_cashier_agent = $parent->createMainCashierAgent();
        }
      }

      $service = new AgentLedgerService();
      $service->createLedgerAccountsForAgent($model);

      //   LedgerService::createCurrencySubAccountsForAgent($model);
    }
  }

  public function updated($model)
  {
    // Check if the type has changed to 'agent' or 'cashier'
    if ($model->wasChanged('type')) {
      $oldType = $model->getOriginal('type');
      $newType = $model->type;

      // If type changed to 'agent' or 'cashier', create ledger accounts
      if ($newType === 'agent'  && $model->isActive()) {
        Log::info("Agent type changed from '{$oldType}' to '{$newType}' for agent {$model->name} (ID: {$model->id}). Creating ledger accounts.");

        $service = new AgentLedgerService();
        $service->createLedgerCashAccountsForAgent($model);

        Log::info("Ledger accounts created successfully for agent {$model->name} after type change.");
      }
    }

    // Check if agent status changed (enabled/disabled)
    if ($model->wasChanged('is_active')) {
      $oldStatus = $model->getOriginal('is_active');
      $newStatus = $model->is_active;

      if (!$newStatus && $oldStatus) {
        // Agent was disabled - logout all associated users immediately
        Log::info("Agent {$model->name} (ID: {$model->id}) was disabled. Dispatching logout job for all associated users.");

        // Dispatch job for immediate logout
        LogoutAgentUsersJob::dispatch($model->id, $model->name);

        // Also perform immediate logout in observer for faster response
        $this->logoutAgentUsers($model);
      } elseif ($newStatus && !$oldStatus) {
        // Agent was enabled
        Log::info("Agent {$model->name} (ID: {$model->id}) was enabled.");
      }
    }

    // Also handle other updates that might need ledger account creation
  }

  /**
   * Logout all users associated with an agent
   */
  private function logoutAgentUsers($agent)
  {
    try {
      // Get all users associated with this agent
      $users = DB::table('users')
        ->where('agent_id', $agent->id)
        ->whereNotNull('agent_id')
        ->get();

      $logoutCount = 0;

      foreach ($users as $user) {
        if ($this->logoutUser($user->id)) {
          $logoutCount++;
        }
      }

      Log::info("Successfully logged out {$logoutCount} users for agent {$agent->name} (ID: {$agent->id})");
    } catch (\Exception $e) {
      Log::error("Error logging out users for agent {$agent->name} (ID: {$agent->id}): " . $e->getMessage());
    }
  }

  /**
   * Logout a specific user from all guards
   */
  private function logoutUser($userId)
  {
    try {
      // Invalidate personal access tokens
      $tokensDeleted = DB::table('personal_access_tokens')
        ->where('tokenable_type', 'App\Models\User')
        ->where('tokenable_id', $userId)
        ->delete();

      // Clear user's cache
      Cache::forget("user_session_{$userId}");
      Cache::forget("agent_session_{$userId}");

      // Clear Redis sessions and user data using helper
      RedisSessionHelper::clearUserSessions($userId);

      Log::info("User ID {$userId} has been logged out successfully");
    } catch (\Exception $e) {
      Log::error("Error logging out user ID {$userId}: " . $e->getMessage());
    }
  }
}
