<?php

namespace App\Livewire\Transfers;

use App\Models\Agent;
use App\Models\Currency;
use App\Models\TransferPackage;
use App\Models\TransferPackageDetail;
use Livewire\Component;

class TransferPackages extends Component
{
    public $agentId;
    public $currency = 'USD';
    public $packages = [];
    public $start = 1;
    public $end = 5000;
    public $step = 500;

    public $baseSendFee = 2;
    public $sendFeeIncrement = 1;

    public $baseReceiveFee = 2;
    public $receiveFeeIncrement = 1;

    public $availableCurrencies = [];
    public $availableAgents = [];
    public  $package = ['name' => '', 'currency' => 'USD', 'agent_id' => null];

    protected function getListeners()
    {
        return [
            'deleteAlert' => 'deleteAlert',
            'editTransferPackage' => 'editPackage',
            'deleteTransferPackage' => 'deletePackage',

        ];
    }
    public function deleteAlert($id)
    {
        $this->dispatch('areyousure', ['msg' => __("alerts.areyousure"), 'emitName' => 'deleteTransferPackage', 'action' => __("actions.delete"), 'title' => __("alerts.delete"), 'data' => ['id' => $id]]);
    }
    public function  deletePackage($data)
    {
        try {
            $package = TransferPackage::find($data['id']);
            $package->delete();
            $this->dispatch("RefreshYajraDatatable", ['table' => 'transferpackage-table']);
            $this->dispatch('sweetalert:success', ['msg' => __("alerts.deleted_successfully"), "title" => __("alerts.success"), "type" => "success"]);
        } catch (\Exception $e) {
            $this->dispatch("sweetalert:error", ['msg' => __('alerts.something_went_wrong'), "title" => "Error", "type" => "error"]);
        }
        $this->dispatch("UnBlockUI");
    }
    public  function editPackage($id)
    {
        $package = TransferPackage::with('details')->find($id);


        if (!$package) {
            $this->reset(['package', 'packages']);
            return;
        }

        $this->package = [
            'id' => $package->id,
            'name' => $package->name,
            'currency' => $package->currency,
            'agent_id' => $this->agentId,
        ];

        $this->packages = $package->details->map(function ($detail) {
            return [
                'id' => $detail->id,
                'min' => $detail->min_amount,
                'max' => $detail->max_amount,
                'send_fee' => $detail->send_fee,
                'receive_fee' => $detail->receive_fee,
                'another_fee' => $detail->another_fee,
            ];
        })->toArray();
    }
    public function mount()
    {
        $this->availableCurrencies = Currency::pluck('name', 'code')->toArray();

        $this->availableAgents = Agent::pluck('name', 'id')->toArray();
        $this->package['currency'] = array_key_first($this->availableCurrencies);

        $this->loadPackage();
    }
    public function generatePackages()
    {
        $this->packages = [];

        $currentMin = $this->start;
        $sendFee = $this->baseSendFee;
        $receiveFee = $this->baseReceiveFee;

        while ($currentMin <= $this->end) {
            $currentMax = min($currentMin + $this->step - 1, $this->end);

            $this->packages[] = [
                'id' => null,
                'min' => $currentMin,
                'max' => $currentMax,
                'send_fee' => $sendFee,
                'receive_fee' => $receiveFee,
                'send_fee_type' => 'fixed',
                'receive_fee_type' => 'fixed',
            ];

            $currentMin = $currentMax + 1;
            $sendFee += $this->sendFeeIncrement;
            $receiveFee += $this->receiveFeeIncrement;
        }
    }

    public function loadPackage()
    {
        if (!$this->agentId) {
            $this->reset(['package', 'packages']);
            return;
        }

        // Get the first package for agent and currency, eager load details
        $package = TransferPackage::whereHas('agents', fn($q) => $q->where('agent_id', $this->agentId))
            ->where('currency', $this->currency)
            ->with('details')
            ->first();

        if (!$package) {
            $this->reset(['package', 'packages']);
            return;
        }

        $this->package = [
            'id' => $package->id,
            'name' => $package->name,
            'currency' => $package->currency,
            'agent_id' => $this->agentId,
        ];

        $this->packages = $package->details->map(function ($detail) {
            return [
                'id' => $detail->id,
                'min' => $detail->min_amount,
                'max' => $detail->max_amount,
                'send_fee' => $detail->send_fee,
                'receive_fee' => $detail->receive_fee,
                'another_fee' => $detail->another_fee,
            ];
        })->toArray();
    }

    public function updatedCurrency()
    {
        $this->loadPackage();
    }
    public function updatedAgentId()
    {
        $this->loadPackage();
    }

    public function addPackage()
    {
        $this->packages[] = ['id' => null, 'min' => null, 'max' => null, 'fee' => null];
    }

    public function removePackage($index)
    {
        if (isset($this->packages[$index]['id'])) {
            TransferPackageDetail::destroy($this->packages[$index]['id']);
        }

        unset($this->packages[$index]);
        $this->packages = array_values($this->packages);
    }


    public function save()
    {
        $this->validate([
            'package.name' => 'required',
            'package.currency' => 'required',
            'packages.*.min' => 'required|numeric|min:0',
            'packages.*.max' => 'required|numeric|gt:packages.*.min',
            'packages.*.send_fee' => 'required|numeric|min:0',
            'packages.*.receive_fee' => 'required|numeric|min:0',
            'packages.*.send_fee_type' => 'required|in:fixed,percent',
            'packages.*.receive_fee_type' => 'required|in:fixed,percent',
        ]);

        if (!isset($this->package['id'])) {
            $package = TransferPackage::create($this->package);
        } else {
            $package = TransferPackage::find($this->package['id']);
            $package->update($this->package);
        }

        $existingIds = $package->details()->pluck('id')->toArray();
        $submittedIds = [];

        foreach ($this->packages as $pkg) {

            $detail = TransferPackageDetail::updateOrCreate(
                ['id' => $pkg['id'] ?? null, 'package_id' => $package->id],
                [
                    'min_amount' => $pkg['min'],
                    'max_amount' => $pkg['max'],
                    'send_fee' => $pkg['send_fee'],
                    'receive_fee' => $pkg['receive_fee'],
                    'send_fee_type' => $pkg['send_fee_type'] ?? 'fixed',
                    'receive_fee_type' => $pkg['receive_fee_type'] ?? 'fixed',
                ]
            );
            $submittedIds[] = $detail->id;
        }

        // Delete details removed in the form
        $toDelete = array_diff($existingIds, $submittedIds);
        if (!empty($toDelete)) {
            TransferPackageDetail::whereIn('id', $toDelete)->delete();
        }

        $this->reset('package', 'packages');

        $this->dispatch("RefreshYajraDatatable", ['table' => 'transferpackage-table']);

        $this->dispatch("sweetalert:success", ['msg' => "Package Saved Successfully", "title" => "Success", "type" => "success"]);
    }

    public function render()
    {
        return view('livewire.transfers.transfer-packages');
    }
}
