<?php

namespace App\Livewire\Shortcut;

use Livewire\Component;

use Livewire\WithFileUploads;
use App\Models\Shortcut;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class Add extends Component
{
    use WithFileUploads;
    public $shortcuts = [];
    public $name, $url, $iconFile;
    public $menuOptions = [];
    public $editId = null;

    public function editShortcut($id)
    {
        $shortcut = Shortcut::findOrFail($id);

        $this->editId = $shortcut->id;
        $this->name = $shortcut->name;
        $this->url = $shortcut->url;
    }

    public function updateShortcut()
    {
        $this->validate([
            'name' => 'required|string',
            'url' => 'required|string|in:' . implode(',', array_keys($this->menuOptions)),
            'iconFile' => 'nullable|image|max:1024',
        ]);
        try {
            $shortcut = Shortcut::findOrFail($this->editId);
            $shortcut->update([
                'name' => $this->name,
                'url' => $this->url,
            ]);

            if ($this->iconFile) {
                $shortcut->clearMediaCollection('icon');
                $shortcut->addMedia($this->iconFile->getRealPath())
                    ->usingFileName($this->iconFile->getClientOriginalName())
                    ->toMediaCollection('icon');
            }
            $this->dispatch("sweetalert:success", ['msg' =>  __('alerts.updated_successfully'), "title" => "Success", "type" => "success"]);
        } catch (\Exception $e) {
            $this->dispatch("sweetalert:error", ['msg' => __('alerts.something_went_wrong'), "title" => "Error", "type" => "error"]);
        }

        $this->reset(['editId', 'name', 'url', 'iconFile']);
    }

    public function deleteShortcut($id)
    {
        $shortcut = Shortcut::findOrFail($id);
        $shortcut->delete();
    }

    public function mount()
    {
        $json = file_get_contents(resource_path('menu/verticalMenu.json'));
        $menu = json_decode($json, true)['menu'] ?? [];
        $this->menuOptions = $this->flattenMenu($menu);
        $this->shortcuts = Shortcut::get();
    }

    protected function flattenMenu(array $items, $prefix = ''): array
    {
        $flat = [];
        foreach ($items as $item) {
            $flat[$item['url']] = $prefix . $item['name'];
            if (!empty($item['submenu'])) {
                $flat += $this->flattenMenu($item['submenu'], $prefix . ' - ');
            }
        }
        return $flat;
    }

    public function addShortcut()
    {
        $this->validate([
            'name' => 'required|string',
            'url' => [
                'required',
                'string',
                'in:' . implode(',', array_keys($this->menuOptions)),
                'unique:shortcuts,url', // ✅ Ensure URL is not duplicated
            ],
            'iconFile' => 'required|image|max:1024',
        ]);
        try {
            $shortcut = Shortcut::create([
                'name' => $this->name,
                'url' => $this->url,

            ]);

            $shortcut->addMedia($this->iconFile->getRealPath())
                ->usingFileName($this->iconFile->getClientOriginalName())
                ->toMediaCollection('icon');

            $this->reset(['name', 'url', 'iconFile']);
            $this->reset(['name', 'url', 'iconFile']);
            $this->dispatch("sweetalert:success", ['msg' =>  __('alerts.created_successfully'), "title" => "Success", "type" => "success"]);
        } catch (\Exception $e) {
            Log::error($e);
            $this->dispatch("sweetalert:error", ['msg' => __('alerts.something_went_wrong'), "title" => "Error", "type" => "error"]);
        }
    }

    public function render()
    {
        return view('livewire.shortcut.add');
    }
}
