<?php

namespace App\Livewire\Settings\Tabs;

use Livewire\Component;
use Rawilk\Settings\Facades\Settings;

class TransferValidation extends Component
{
  public $transferValidation = [
    // Beneficiary name word limits
    'beneficiary_name_min_words' => 2,
    'beneficiary_name_max_words' => 10,

    // Phone number character limits
    'phone_number_min_chars' => 8,
    'phone_number_max_chars' => 15,

    // Additional validation settings
    'require_beneficiary_name' => true,
    'require_phone_number' => true,
    'allow_special_chars_in_name' => false,
    'phone_number_format' => 'international', // international, local, flexible
  ];

  public $phoneFormats = [
    'international' => 'International Format (+966501234567)',
    'local' => 'Local Format (0501234567)',
    'flexible' => 'Flexible (Any format)'
  ];

  public function mount()
  {
    // Load saved settings with defaults
    foreach ($this->transferValidation as $key => $default) {
      $this->transferValidation[$key] = Settings::get("transfer_validation.$key", $default);
    }
  }

  public function saveTransferValidationSettings()
  {
    // Validate the settings
    $this->validate([
      'transferValidation.beneficiary_name_min_words' => 'required|integer|min:1|max:50',
      'transferValidation.beneficiary_name_max_words' => 'required|integer|min:1|max:100',
      'transferValidation.phone_number_min_chars' => 'required|integer|min:5|max:20',
      'transferValidation.phone_number_max_chars' => 'required|integer|min:5|max:25',
    ], [
      'transferValidation.beneficiary_name_min_words.required' => 'الحد الأدنى لعدد الكلمات مطلوب',
      'transferValidation.beneficiary_name_min_words.integer' => 'الحد الأدنى لعدد الكلمات يجب أن يكون رقماً صحيحاً',
      'transferValidation.beneficiary_name_min_words.min' => 'الحد الأدنى لعدد الكلمات يجب أن يكون 1 على الأقل',
      'transferValidation.beneficiary_name_min_words.max' => 'الحد الأدنى لعدد الكلمات يجب أن يكون 50 كحد أقصى',

      'transferValidation.beneficiary_name_max_words.required' => 'الحد الأقصى لعدد الكلمات مطلوب',
      'transferValidation.beneficiary_name_max_words.integer' => 'الحد الأقصى لعدد الكلمات يجب أن يكون رقماً صحيحاً',
      'transferValidation.beneficiary_name_max_words.min' => 'الحد الأقصى لعدد الكلمات يجب أن يكون 1 على الأقل',
      'transferValidation.beneficiary_name_max_words.max' => 'الحد الأقصى لعدد الكلمات يجب أن يكون 100 كحد أقصى',

      'transferValidation.phone_number_min_chars.required' => 'الحد الأدنى لعدد الأحرف في رقم الهاتف مطلوب',
      'transferValidation.phone_number_min_chars.integer' => 'الحد الأدنى لعدد الأحرف في رقم الهاتف يجب أن يكون رقماً صحيحاً',
      'transferValidation.phone_number_min_chars.min' => 'الحد الأدنى لعدد الأحرف في رقم الهاتف يجب أن يكون 5 على الأقل',
      'transferValidation.phone_number_min_chars.max' => 'الحد الأدنى لعدد الأحرف في رقم الهاتف يجب أن يكون 20 كحد أقصى',

      'transferValidation.phone_number_max_chars.required' => 'الحد الأقصى لعدد الأحرف في رقم الهاتف مطلوب',
      'transferValidation.phone_number_max_chars.integer' => 'الحد الأقصى لعدد الأحرف في رقم الهاتف يجب أن يكون رقماً صحيحاً',
      'transferValidation.phone_number_max_chars.min' => 'الحد الأقصى لعدد الأحرف في رقم الهاتف يجب أن يكون 5 على الأقل',
      'transferValidation.phone_number_max_chars.max' => 'الحد الأقصى لعدد الأحرف في رقم الهاتف يجب أن يكون 25 كحد أقصى',
    ]);

    // Ensure min is less than max
    if ($this->transferValidation['beneficiary_name_min_words'] >= $this->transferValidation['beneficiary_name_max_words']) {
      $this->addError('transferValidation.beneficiary_name_min_words', 'الحد الأدنى لعدد الكلمات يجب أن يكون أقل من الحد الأقصى');
      return;
    }

    if ($this->transferValidation['phone_number_min_chars'] >= $this->transferValidation['phone_number_max_chars']) {
      $this->addError('transferValidation.phone_number_min_chars', 'الحد الأدنى لعدد الأحرف في رقم الهاتف يجب أن يكون أقل من الحد الأقصى');
      return;
    }

    // Save all settings
    foreach ($this->transferValidation as $key => $value) {
      Settings::set("transfer_validation.$key", $value);
    }

    $this->dispatch("sweetalert:success", [
      'msg' => 'تم حفظ إعدادات التحقق من التحويلات بنجاح',
      'title' => 'تم الحفظ',
      'icon' => 'success'
    ]);
  }

  public function resetToDefaults()
  {
    $defaults = [
      'beneficiary_name_min_words' => 2,
      'beneficiary_name_max_words' => 10,
      'phone_number_min_chars' => 8,
      'phone_number_max_chars' => 15,
      'require_beneficiary_name' => true,
      'require_phone_number' => true,
      'allow_special_chars_in_name' => false,
      'phone_number_format' => 'international',
    ];

    $this->transferValidation = $defaults;

    $this->dispatch("sweetalert:success", [
      'msg' => 'تم إعادة تعيين الإعدادات إلى القيم الافتراضية',
      'title' => 'تم الإعادة',
      'icon' => 'info'
    ]);
  }

  public function render()
  {
    return view('livewire.settings.tabs.transfer-validation-settings');
  }
}
