<?php

namespace App\Livewire\Settings\Tabs;

use Livewire\Component;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Rawilk\Settings\Facades\Settings;

class ReferenceNumber extends Component
{
  public $referenceSettings = [
    'prefix' => 'REF',
    'branch_code_length' => 3,
    'transaction_id_length' => 6,
    'separator' => '-',
    'auto_increment' => false,
    'current_sequence' => 1,
    'reset_daily' => false,
    'custom_format' => '{prefix}{separator}{branch_code}{separator}{transaction_id}'
  ];

  public $showPreview = false;
  public $previewReference = '';

  protected $rules = [
    'referenceSettings.prefix' => 'string|max:10',
    'referenceSettings.branch_code_length' => 'required|integer|min:1|max:10',
    'referenceSettings.transaction_id_length' => 'required|integer|min:1|max:10',
    'referenceSettings.separator' => 'required|string|max:5',
    'referenceSettings.auto_increment' => 'boolean',
    'referenceSettings.current_sequence' => 'required|integer|min:1',
    'referenceSettings.reset_daily' => 'boolean',
    'referenceSettings.custom_format' => 'required|string|max:100'
  ];

  public function mount()
  {
    $this->loadSettings();
  }

  public function loadSettings()
  {
    $settings = Cache::get('reference_number_settings');
    if ($settings) {
      $this->referenceSettings = array_merge($this->referenceSettings, $settings);
    }
  }

  public function saveSettings()
  {
    $this->validate();

    // Save to cache
    Cache::put('reference_number_settings', $this->referenceSettings, now()->addYear());

    // Save to database if you want persistence
    $this->saveToDatabase();

    $this->dispatch('toast', [
      'type' => 'success',
      'message' => 'Reference number settings saved successfully!'
    ]);
  }

  private function saveToDatabase()
  {
    // Create or update settings in database
    Settings::set('reference_number_settings', $this->referenceSettings);
  }





  public function generatePreview()
  {
    $this->previewReference = $this->generateReferenceNumber();
    $this->showPreview = true;
  }

  public function generateReferenceNumber()
  {
    $settings = $this->referenceSettings;

    if ($settings['auto_increment']) {
      $sequence = $this->getNextSequence();
      $transactionId = str_pad($sequence, $settings['transaction_id_length'], '0', STR_PAD_LEFT);
    } else {
      $transactionId = str_pad(mt_rand(0, pow(10, $settings['transaction_id_length']) - 1), $settings['transaction_id_length'], '0', STR_PAD_LEFT);
    }

    $branchCode = str_pad(mt_rand(0, pow(10, $settings['branch_code_length']) - 1), $settings['branch_code_length'], '0', STR_PAD_LEFT);

    $format = $settings['custom_format'];
    $reference = str_replace(
      ['{prefix}', '{separator}', '{branch_code}', '{transaction_id}'],
      [$settings['prefix'], $settings['separator'], $branchCode, $transactionId],
      $format
    );

    return $reference;
  }

  private function getNextSequence()
  {
    $key = 'reference_sequence_' . date('Y-m-d');

    if ($this->referenceSettings['reset_daily']) {
      $sequence = Cache::increment($key);
    } else {
      $sequence = Cache::increment('reference_sequence');
    }

    return $sequence;
  }

  public function resetSequence()
  {
    if ($this->referenceSettings['auto_increment']) {
      if ($this->referenceSettings['reset_daily']) {
        Cache::forget('reference_sequence_' . date('Y-m-d'));
      } else {
        Cache::forget('reference_sequence');
      }

      $this->referenceSettings['current_sequence'] = 1;

      $this->dispatch('toast', [
        'type' => 'success',
        'message' => 'Sequence reset successfully!'
      ]);
    }
  }

  public function render()
  {
    return view('livewire.settings.tabs.reference-number');
  }
}
