<?php

namespace App\Livewire\Ledger;

use App\Models\CustomLedgerAccount;
use App\Models\BalanceOverwrite;
use App\Models\Currency;
use Livewire\Component;

class ModernAccountTreeNode extends Component
{
    public $account;
    public $level;
    public $search;
    public $expandedNodes = [];
    public $expanded = false;
    public $children = [];
    public $balances = [];

    protected $listeners = [
        'refreshTree' => '$refresh',
        'expandAll' => 'expand',
        'collapseAll' => 'collapse',
    ];

    public function mount($account, $level = 0, $search = '', $expandedNodes = [])
    {
        $this->account = $account;
        $this->level = $level;
        $this->search = $search;
        $this->expandedNodes = $expandedNodes;

        // Check if this node should be expanded
        $this->expanded = in_array($this->account->ledgerUuid, $this->expandedNodes);

        // If expanded, load children
        if ($this->expanded) {
            $this->loadChildren();
        }

        $this->loadBalances();
    }

    public function toggle()
    {
        if ($this->expanded) {
            $this->expanded = false;
            $this->children = [];
        } else {
            $this->loadChildren();
            $this->expanded = true;
        }
    }

    public function expand()
    {
        if (!$this->expanded) {
            $this->loadChildren();
            $this->expanded = true;
        }

        // Also expand all children
        $this->dispatch('expandAll');
    }

    public function collapse()
    {
        $this->expanded = false;
        $this->children = [];

        // Also collapse all children
        $this->dispatch('collapseAll');
    }

    public function loadChildren()
    {
        $this->children = CustomLedgerAccount::with(['names', 'children'])
            ->where('parentUuid', $this->account->ledgerUuid)
            ->get();
    }

    public function loadBalances()
    {
        $this->balances = BalanceOverwrite::where('ledgerUuid', $this->account->ledgerUuid)
            ->get()
            ->mapWithKeys(function ($balance) {
                return [$balance->currency => (float) $balance->balance];
            });
    }

    public function getAccountNameProperty()
    {
        return $this->account->names->firstWhere('language', app()->getLocale())?->name
            ?? $this->account->names->first()?->name
            ?? $this->account->code;
    }

    public function getHasChildrenProperty()
    {
        return CustomLedgerAccount::where('parentUuid', $this->account->ledgerUuid)->exists();
    }

    public function getChildrenCountProperty()
    {
        return CustomLedgerAccount::where('parentUuid', $this->account->ledgerUuid)->count();
    }

    public function getTotalBalanceProperty()
    {
        return collect($this->balances)->sum();
    }

    public function getAccountTypeProperty()
    {
        if ($this->account->category) {
            return 'category';
        }

        if ($this->account->debit) {
            return 'debit';
        }

        if ($this->account->credit) {
            return 'credit';
        }

        return 'unknown';
    }

    public function getAccountTypeColorProperty()
    {
        return match ($this->accountType) {
            'category' => 'bg-blue-100 text-blue-800',
            'debit' => 'bg-green-100 text-green-800',
            'credit' => 'bg-red-100 text-red-800',
            default => 'bg-gray-100 text-gray-800',
        };
    }

    public function getAccountTypeIconProperty()
    {
        return match ($this->accountType) {
            'category' => 'ti ti-folder',
            'debit' => 'ti ti-arrow-down',
            'credit' => 'ti ti-arrow-up',
            default => 'ti ti-file',
        };
    }

    public function addChildAccount()
    {
        $this->dispatch('open_modal', (object) [
            'action' => 'show',
            'id' => 'ledgerAccountModal',
            'data' => (object) [
                'mode' => 'create',
                'parentUuid' => $this->account->ledgerUuid,
                'agentId' => null,
            ]
        ]);
    }

    public function editAccount()
    {
        $this->dispatch('open_modal', (object) [
            'action' => 'show',
            'id' => 'ledgerAccountModal',
            'data' => (object) [
                'mode' => 'edit',
                'ledgerUuid' => $this->account->ledgerUuid,
            ]
        ]);
    }

    public function deleteAccount()
    {
        $this->dispatch('deleteAccount', $this->account->ledgerUuid);
    }

    public function viewBalances()
    {
        $this->dispatch('viewAccountBalances', $this->account->ledgerUuid);
    }

    public function render()
    {
        return view('livewire.ledger.modern-account-tree-node');
    }
}
