<?php

namespace App\Livewire\Exchange;

use App\Models\Currency;
use App\Models\ExchangePrice;
use Illuminate\Support\Facades\Event;
use Livewire\Component;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Validator;
use OwenIt\Auditing\Events\AuditCustom;

class Prices extends Component
{
  public $from_currency = 'USD';
  public $to_currency = 'TRY';
  public $price_source = 'static';
  public $factor = 'multiply';
  public $static_price;
  public $api_key;
  public $difference;
  public $operation = '*';
  public $type = 'balance';
  public $active_for_customers = true;
  public  $currencies = [];

  public $editingId = null;

  public $balancePrices = [];
  public $transferPrices = [];

  protected function  getListeners()
  {
    return [
      "SetExchange" => "setExchange",
      "DeletePrice" => "DeletePrice"
    ];
  }
  public function setExchange($data)
  {

    $from = $data['from'];
    $to = $data['to'];
    $rate = $data['rate'];
    $apiKey = $data['apiKey'];
    $this->from_currency = $from;
    $this->to_currency = $to;
    $this->operation = $data['factor'] == "multiply" ? "*" : "/";

    if ($this->price_source === 'static') {
      $this->static_price = $rate;
    } else {
      $this->difference = 0;
      $this->api_key = $apiKey;
    }
  }
  public function mount()
  {
    $this->currencies = Currency::get();

    $this->loadPrices();
  }



  public function submit()
  {
    $this->validate($this->rules());

    ExchangePrice::disableAuditing();

    // Check if this is a create or update operation
    $isUpdate = !empty($this->editingId);

    $row = ExchangePrice::updateOrCreate([
      'id' => $this->editingId,
    ], [
      'from_currency' => $this->from_currency,
      'to_currency' => $this->to_currency,
      'price_source' => $this->price_source,
      'static_price' => $this->price_source === 'static' ? $this->static_price : null,
      'api_key' => $this->price_source === 'api' ? $this->api_key : null,
      'difference' => $this->price_source === 'api' ? $this->difference : null,
      'operation' => $this->operation,
      'type' => $this->type,
      'active_for_customers' => $this->active_for_customers,
    ]);

    ExchangePrice::enableAuditing();

    // Set audit event based on operation type
    $row->auditEvent = $isUpdate ? 'Update_Price' : 'Create_Price';
    $row->isCustomEvent = true;

    if ($isUpdate) {
      // For updates: set old and new values
      $row->auditCustomOld = [
        'from_currency' => $row->from_currency,
        'to_currency' => $row->to_currency,
        'price_source' => $row->price_source,
        'static_price' => $row->static_price,
        'api_key' => $row->api_key,
        'difference' => $row->difference,
        'operation' => $row->operation,
        'type' => $row->type,
        'active_for_customers' => $row->active_for_customers,
      ];
      $row->auditCustomNew = [
        'from_currency' => $this->from_currency,
        'to_currency' => $this->to_currency,
        'price_source' => $this->price_source,
        'static_price' => $this->static_price,
        'api_key' => $this->api_key,
        'difference' => $this->difference,
        'operation' => $this->operation,
        'type' => $this->type,
        'active_for_customers' => $this->active_for_customers,
      ];
    } else {
      // For creates: only set new values, old values are null
      $row->auditCustomOld = [];
      $row->auditCustomNew = [
        'from_currency' => $this->from_currency,
        'to_currency' => $this->to_currency,
        'price_source' => $this->price_source,
        'static_price' => $this->static_price,
        'api_key' => $this->api_key,
        'difference' => $this->difference,
        'operation' => $this->operation,
        'type' => $this->type,
        'active_for_customers' => $this->active_for_customers,
      ];
    }

    Event::dispatch(new AuditCustom($row));
    $this->resetForm();
    $this->loadPrices();
  }

  public function edit($id)
  {
    $row = ExchangePrice::findOrFail($id);

    $this->editingId = $row->id;
    $this->from_currency = $row->from_currency;
    $this->to_currency = $row->to_currency;
    $this->price_source = $row->price_source;
    $this->static_price = $row->static_price;
    $this->api_key = $row->api_key;
    $this->difference = $row->difference;
    $this->operation = $row->operation;
    $this->type = $row->type;
    $this->active_for_customers = $row->active_for_customers;
  }
  public  function DeletePrice($data)
  {
    ExchangePrice::disableAuditing();
    $row = ExchangePrice::findOrFail($data['id']);
    ExchangePrice::findOrFail($data['id'])->delete();
    ExchangePrice::enableAuditing();
    $row->auditEvent = 'Delete_Price';
    $row->isCustomEvent = true;
    $row->auditCustomOld = [];
    $row->auditCustomNew = [];
    Event::dispatch(new AuditCustom($row));
    $this->loadPrices();
    $this->dispatch("UnBlockUI");
  }

  public function delete($id)
  {
    $this->dispatch("areyousure", [
      'msg' => __("Are you sure you want to delete this price?"),
      'emitName' => 'DeletePrice',
      'action' => __("Delete"),
      'title' => __("Delete"),
      'data' => ['id' => $id]
    ]);
  }

  public function resetForm()
  {
    $this->reset(['editingId', 'from_currency', 'to_currency', 'price_source', 'static_price', 'api_key', 'difference', 'operation', 'type', 'active_for_customers']);
    $this->from_currency = 'USD';
    $this->to_currency = 'TRY';
    $this->price_source = 'static';
    $this->operation = '*';
    $this->type = 'balance';
    $this->active_for_customers = true;
  }

  public function loadPrices()
  {
    $this->balancePrices = ExchangePrice::where('type', 'balance')->get();
    $this->transferPrices = ExchangePrice::where('type', 'transfer')->get();
  }

  public function rules()
  {
    return [
      'from_currency' => [
        'required',
        Rule::unique('exchange_prices')
          ->where(function ($query) {
            return $query->where('to_currency', $this->to_currency);
          })
          ->where(function ($query) {
            return $query->where('type', $this->type);
          })
          ->ignore($this->editingId),
      ],
      'to_currency' => ['required'],
      'price_source' => ['required', Rule::in(['static', 'api'])],
      'static_price' => ['required_if:price_source,static', 'nullable', 'numeric', 'min:0.01'],
      'api_key' => ['required_if:price_source,api', 'nullable', 'string'],
      'difference' => ['nullable', 'numeric'],
      'operation' => ['required_if:price_source,api', Rule::in(['*', '/'])],
      'type' => ['required', Rule::in(['balance', 'transfer'])],
      'active_for_customers' => ['boolean'],
    ];
  }
  public function render()
  {
    return view('livewire.exchange.prices');
  }
}
