<?php

namespace App\Livewire\Exchange;

use App\Exceptions\InvalidExchangeRateException;
use App\Models\Agent;
use App\Models\Currency;
use App\Models\Exchange;
use App\Models\LedgerExchange;
use App\Models\LedgerTransfer;
use App\Services\Transfer\ExchangeService;
use Illuminate\Support\Facades\Log;
use Livewire\Component;
use App\Services\ExchangeRates\CurrencyConverter;
use Illuminate\Support\Facades\DB;
use OwenIt\Auditing\Events\AuditCustom;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\Facades\Gate;

class Add extends Component
{
  public $exchange = ['currency' => null, 'delivery_currency' => null, 'rate' => 1,   'amount' => null, 'delivery_amount' => null, 'type' => "exchange", 'rate_factor' => "multiply", 'status' => "completed"];
  public $agents = [];
  public  $currencies = [];
  public string $createExchangeMessage = '';
  public bool $canCreateExchange = false;
  public $lastEditedField = null;
  protected  function  getListeners()
  {
    return [
      'SetExchangeAgent' => 'SetExchangeAgent',
      'SetExchange' => 'setExchange'
    ];
  }
  public function setExchange($data)
  {
    Log::info("Setting exchange with data: ", $data);
    $this->exchange['currency'] = $data['from'];
    $this->exchange['delivery_currency'] = $data['to'];
    $this->exchange['rate'] = $data['rate'];
    $this->exchange['rate_factor'] = $data['factor'] == "/" ? "divide" : "multiply";

    $this->Calculate();
  }
  public  function  UpdatedExchangeCurrency()
  {
    $this->GetRateandFactor();
  }
  public  function  UpdatedExchangeDeliveryCurrency()
  {
    $this->GetRateandFactor();
  }
  public  function  GetRateandFactor()
  {
    try {
      $from_amount = $this->exchange['amount'] ?? 0;
      $from_currency = $this->exchange['currency'];
      $to_currency = $this->exchange['delivery_currency'];

      if (!$to_currency || !$from_currency) {

        $this->exchange['rate_factor'] = "multiply";
        $this->exchange['rate'] = 1;
        return;
      }
      if ($from_currency == $to_currency) {
        $this->exchange['rate_factor'] = "multiply";
        $this->exchange['rate'] = 1;
        return;
      }
      $factor = getCurrencyFactor($from_currency, $to_currency, 'transfer');
      $this->exchange['rate_factor'] = $factor;
      $rate = exchange_rate($from_currency, $to_currency,   'transfer');
      $this->exchange['rate'] = $rate;
    } catch (InvalidExchangeRateException $e) {
      $this->dispatch("sweetalert:error", ['msg' => $e->getMessage(), 'title' => 'Error']);
    } catch (\Exception $e) {
      $this->dispatch("sweetalert:error", ['msg' => $e->getMessage(), 'title' => 'Error']);
    }
  }
  public  function CheckExchangeCurrencyBalance() {}

  public function  SetExchangeAgent($data)
  {


    $this->exchange['sender_id'] = (int) $data['agent_id'];
    $this->exchange['receiver_id'] = (int) $data['agent_id'];
  }
  public function mount()
  {
    $this->agents = LedgerExchange::GetSourceAgents();;
    $this->currencies = Currency::get();

    $response = Gate::inspect('create', \App\Models\LedgerExchange::class);

    $this->canCreateExchange = $response->allowed();
    $this->createExchangeMessage = $response->message() ?? '';
  }



  public function updated($propertyName)
  {
    // if (isset($this->exchange['currency']) && isset($this->exchange['delivery_currency'])) {
    //     $this->exchange['rate_factor'] = getCurrencyFactor($this->exchange['currency'], $this->exchange['delivery_currency'], 'balance');
    // }

    if (str_contains($propertyName, 'amount')) {
      $this->lastEditedField = $propertyName;
    }

    $this->Calculate();
  }
  public function Calculate()
  {
    $from_amount = $this->exchange['amount'] ?? 0;
    $to_amount = $this->exchange['delivery_amount'] ?? 0;

    // If both are empty or only one has value, return
    // if ((!$from_amount && !$to_amount) || ($from_amount && $to_amount)) {
    //     return;
    // }

    $converter = new CurrencyConverter();

    if ($this->lastEditedField != 'exchange.delivery_amount') {
      $calculated_amount = $converter->convert(
        $from_amount,
        $this->exchange['currency'],
        $this->exchange['delivery_currency'],
        $this->currencies,
        'balance',
        $this->exchange['rate'],
        $this->exchange['rate_factor'] == 'multiply' ? '*' : '/'
      );

      Log::info("Calculated amount: " . $calculated_amount);
      Log::info($this->exchange);
      $this->exchange['delivery_amount'] = $calculated_amount;
    }
    if ($this->lastEditedField === 'exchange.delivery_amount' && $to_amount) {
      $calculated_amount = $converter->convert(
        $to_amount,
        $this->exchange['delivery_currency'],
        $this->exchange['currency'],
        $this->currencies,
        'balance',
        $this->exchange['rate'],
        $this->exchange['rate_factor'] == 'multiply' ? '/' : '*' // Reverse the operation
      );
      $this->exchange['amount'] = $calculated_amount;
    }
  }




  public function save()
  {
    Log::info("START save()");
    if (!isset($this->exchange['sender_id'])) {
      $this->dispatch("sweetalert:error", ['msg' => __('exchange.select_sender'), 'title' => 'Error']);
      return;
    }
    $rules = [
      'exchange.currency' => 'required|different:exchange.delivery_currency',
      'exchange.delivery_currency' => 'required|different:exchange.currency',

      'exchange.receiver_id' => 'nullable',

      'exchange.amount' => 'required|numeric|min:0.01',
      'exchange.delivery_amount' => 'required|numeric|min:0.01',

    ];
    $this->validate($rules);
    $this->exchange['reference'] = generateReferenceNumber("EXG");

    try {
      DB::beginTransaction();
      LedgerTransfer::disableAuditing();
      Log::info("Creating exchange");
      $ex = LedgerTransfer::create($this->exchange);

      Log::info("Calling ExchangeService");
      $service = new ExchangeService();
      $service->InsertExchangeEntries($ex);

      DB::commit();

      LedgerTransfer::enableAuditing();
      $ex->auditEvent = 'Create_Exchange';
      $ex->isCustomEvent = true;
      $ex->auditCustomOld = [];

      $ex->auditCustomNew = [
        "sender" => $ex->sender->name ?? "",
        "receiver" => $ex->receiver->name ?? "",
        "amount" => $ex->amount,
        "receiver_fee" => $ex->receiver_fee,
        "sender_fee" => $ex->sender_fee,
        "delivery_amount" => $ex->delivery_amount,
        "currency" => $ex->currency,
        "delivery_currency" => $ex->delivery_currency,
        "status" => "Completed"
      ];

      // Step 6: Dispatch the custom audit event
      Event::dispatch(new AuditCustom($ex));
      $ex->disableAuditing();

      $this->dispatch("RefreshYajraDatatable", ['table' => 'exchange-table']);
      $this->dispatch("sweetalert:success", [
        'msg' => __('exchange.exchange_added_successfully'),
        'title' => 'Success'
      ]);
    } catch (InvalidExchangeRateException $e) {
      DB::rollBack();
      Log::warning("Exchange failed due to invalid rate: " . $e->getMessage());

      $this->dispatch("sweetalert:error", [
        'msg' => $e->getMessage(),
        'title' => 'Invalid Rate'
      ]);
    } catch (\Throwable $e) {
      DB::rollBack();
      Log::error('Exchange Save Failed: ' . $e->getMessage());

      $this->dispatch("sweetalert:error", [
        'msg' => 'Exchange failed: ' . $e->getMessage(),
        'title' => 'Error'
      ]);
    } catch (\Exception $e) {
      DB::rollBack();
      Log::error('Exchange Save Failed: ' . $e->getMessage());

      $this->dispatch("sweetalert:error", [
        'msg' => 'Exchange failed: ' . $e->getMessage(),
        'title' => 'Error'
      ]);
    }
  }


  public function getAmountInWords($amount, $currency)
  {
    if ($amount && $currency) {
      return toCurrencyWords(
        floatval($amount),
        $currency,
        app()->getLocale()
      );
    }
    return '';
  }
  public function render()
  {
    return view('livewire.exchange.add');
  }
}
