<?php

namespace App\Livewire\Dashboard;

use App\Models\CustomLedgerAccount;
use App\Models\LedgerTransfer;
use Livewire\Component;
use Illuminate\Support\Facades\Log;


class Dashboard extends Component
{

  public  $statistics = [];
  public  $currencyStatistics = [];
  public function loadStatistics()
  {
    $baseQuery = LedgerTransfer::transfer();
    $feeQuery = LedgerTransfer::transfer();

    $this->statistics = [
      // Status-based
      'total' => (clone $baseQuery)->count(),
      'completed' => (clone $baseQuery)->where('status', 'completed')->count(),
      'pending' => (clone $baseQuery)->where('status', 'pending')->count(),
      'cancelled' => LedgerTransfer::onlyTrashed()->count(),
      'trashed_transfers' => LedgerTransfer::wherein('type', ['transfer', 'approval'])->onlyTrashed()->count(),
      'trashed_bookings' => LedgerTransfer::wherein('type', ['booking', 'exchange'])->onlyTrashed()->count(),

      // Time-based
      'today' => (clone $baseQuery)->whereDate('created_at', today())->count(),
      'month' => (clone $baseQuery)->whereMonth('created_at', now()->month)->count(),

      // Types
      'transfer' => (clone $baseQuery)->where('type', 'transfer')->count(),
      'approval' => (clone $baseQuery)->where('type', 'approval')->count(),
      'booking' => (clone $baseQuery)->where('type', 'booking')->count(),

      // Internal / External
      'internal' => (clone $baseQuery)->whereNull('external_agent_id')->count(),
      'external' => (clone $baseQuery)->whereNotNull('external_agent_id')->count(),

      // Delivered / Undelivered
      'delivered' => (clone $baseQuery)->whereNotNull('delivered_at')->count(),
      'undelivered' => (clone $baseQuery)->whereNull('delivered_at')->count(),





      // High-value
      'count_high_value' => (clone $baseQuery)->where('amount', '>', 10000)->count(),
    ];
  }

  public function loadCurrencyStatistics()
  {
    // Get all active currencies
    $currencies = \App\Models\Currency::get();

    foreach ($currencies as $currency) {
      $baseQuery = LedgerTransfer::transfer()->where('currency', $currency->code);

      $today_baseQuery = LedgerTransfer::transfer()->where('currency', $currency->code)->whereDate('created_at', today());
      // Get currency position data (similar to DataTable logic)
      if ($currency->code != "USD") {
        $currencyPosition = $this->getCurrencyPosition($currency->code);
      }
      $exchangeCost = $this->getExchangeCost($currency->code);

      $this->currencyStatistics[$currency->code] = [
        'currency_name' => $currency->name,
        'currency_symbol' => $currency->symbol,
        'currency_color' => $currency->color ?? '#685dd8',
        'currency_image' => getCurrencyAvatar($currency->code),

        // Status-based
        'total' => (clone $today_baseQuery)->count(),
        'completed' => (clone $today_baseQuery)->where('status', 'completed')->count(),
        'pending' => (clone $today_baseQuery)->where('status', 'pending')->count(),
        'cancelled' => (clone $today_baseQuery)->onlyTrashed()->count(),

        // Time-based
        'today' => (clone $baseQuery)->whereDate('created_at', today())->count(),
        'month' => (clone $baseQuery)->whereMonth('created_at', now()->month)->count(),

        // Types
        'transfer' => (clone $today_baseQuery)->where('type', 'transfer')->count(),
        'approval' => (clone $today_baseQuery)->where('type', 'approval')->count(),
        'booking' => (clone $today_baseQuery)->where('type', 'booking')->count(),

        // Internal / External
        'internal' => (clone $today_baseQuery)->whereNull('external_agent_id')->count(),
        'external' => (clone $today_baseQuery)->whereNotNull('external_agent_id')->count(),

        // Delivered / Undelivered
        'delivered' => (clone $today_baseQuery)->whereNotNull('delivered_at')->count(),
        'undelivered' => (clone $today_baseQuery)->whereNull('delivered_at')->count(),

        // Aggregates
        'total_amount' => (clone $today_baseQuery)->sum('amount'),
        'delivery_amount' => (clone $today_baseQuery)->sum('delivery_amount'),
        'sender_fee' => (clone $today_baseQuery)->sum('sender_fee'),
        'receiver_fee' => (clone $today_baseQuery)->sum('receiver_fee'),
        'send_fee' => (clone $today_baseQuery)->sum('send_fee'),

        // High-value
        'count_high_value' => (clone $baseQuery)->where('amount', '>', 10000)->count(),

        // Currency Position Data (from DataTable logic)
        'currency_amount' => $currencyPosition['amount'] ?? 0,
        'base_amount' => $currencyPosition['base_amount'] ?? 0,
        'exchange_cost' => $exchangeCost,
        'position_color' => $currencyPosition['color'] ?? 'text-blue-600',
      ];
    }
  }

  /**
   * Get currency position data similar to DataTable logic
   */
  private function getCurrencyPosition($currencyCode)
  {
    try {
      // Get the currency account from Abivia Ledger
      $account = CustomLedgerAccount::where('extra->currency_code', $currencyCode)
        ->where('category', 0)
        ->first();

      if (!$account) {
        return [
          'amount' => 0,
          'base_amount' => 0,
          'color' => 'text-red-600',
          'currency_image' => getCurrencyAvatar($currencyCode)
        ];
      }

      $extra = is_string($account->extra) ? json_decode($account->extra, true) : $account->extra;
      $currencyCodeFromExtra = $extra['currency_code'] ?? null;

      if (!$currencyCodeFromExtra) {
        return [
          'amount' => 0,
          'base_amount' => 0,
          'color' => 'text-red-600',
          'currency_image' => getCurrencyAvatar($currencyCode)
        ];
      }

      // Get currency balance
      $balance = $account->balances->firstWhere('currency', $currencyCodeFromExtra);
      $currencyAmount = $balance ? round(-1 * $balance->balance, 0) : 0;
      $currencyColor = ($balance && -1 * $balance->balance < 0) ? 'text-red-600' : 'text-blue-600';

      // Get base amount (USD)
      $baseBalance = $account->balances->firstWhere('currency', 'USD');
      $baseAmount = $baseBalance ? round(-1 * $baseBalance->balance, 0) : 0;

      return [
        'amount' => $currencyAmount,
        'base_amount' => $baseAmount,
        'color' => $currencyColor,
        'currency_image' => getCurrencyAvatar($currencyCode)
      ];
    } catch (\Exception $e) {
      Log::error("Error getting currency position for {$currencyCode}: " . $e->getMessage());
      return [
        'amount' => 0,
        'base_amount' => 0,
        'color' => 'text-red-600',
        'currency_image' => getCurrencyAvatar($currencyCode)
      ];
    }
  }

  /**
   * Get exchange cost per unit
   */
  private function getExchangeCost($currencyCode)
  {
    try {
      if (function_exists('getExchangeCostPerUnitForTable')) {
        return getExchangeCostPerUnitForTable("USD", $currencyCode);
      }
      return 0;
    } catch (\Exception $e) {
      Log::error("Error getting exchange cost for {$currencyCode}: " . $e->getMessage());
      return 0;
    }
  }
  public function LoadGraphData()
  {


    // Donut chart: Total amount by type
    $donut = LedgerTransfer::selectRaw('type, SUM(amount) as total')
      ->groupBy('type')
      ->get();

    $donutLabels = $donut->pluck('type')->map(fn($type) => __($type))->toArray();
    $donutData = $donut->pluck('total')->map(fn($v) => round(floatval($v), 2))->toArray();

    $this->dispatch('initCategoryChart', [
      'labels' => $donutLabels,
      'series' => $donutData,
    ]);
  }



  public  function mount()
  {
    $this->loadStatistics();
    $this->loadCurrencyStatistics();
    $this->LoadGraphData();
  }
  public function render()
  {
    return view('livewire.dashboard.dashboard');
  }
}
