<?php

namespace App\Livewire\Components;

use Livewire\Component;
use App\Models\LedgerTransfer;
use Illuminate\Support\Facades\Log;

class TransferSearch extends Component
{
  public $searchReference = '';
  public $transfer = null;
  public $showResults = false;
  public $errorMessage = '';

  protected $rules = [
    'searchReference' => 'required|min:3|max:50'
  ];

  protected $messages = [
    'searchReference.required' => 'Please enter a transfer reference',
    'searchReference.min' => 'Reference must be at least 3 characters',
    'searchReference.max' => 'Reference must not exceed 50 characters'
  ];

  public function searchTransfer()
  {
    $this->validate();
    $this->errorMessage = '';
    $this->transfer = null;
    $this->showResults = false;

    try {
      // Search for transfer by reference
      $transfer = LedgerTransfer::with([
        'sender',
        'sender.address',
        'address'
      ])->where('reference', 'LIKE', '%' . $this->searchReference . '%')
        ->orWhere('secret', 'LIKE', '%' . $this->searchReference . '%')
        ->first();

      if ($transfer) {
        $this->transfer = $transfer;
        $this->showResults = true;
        Log::info('Transfer found', ['reference' => $this->searchReference, 'transfer_id' => $transfer->id]);
      } else {
        $this->transfer = null;
        $this->errorMessage = 'No transfer found with the provided reference.';
        Log::info('Transfer not found', ['reference' => $this->searchReference]);
      }
    } catch (\Exception $e) {
      $this->transfer = null;
      $this->errorMessage = 'An error occurred while searching. Please try again.';
      Log::error('Transfer search error', ['reference' => $this->searchReference, 'error' => $e->getMessage()]);
    }
  }

  public function clearSearch()
  {
    $this->searchReference = '';
    $this->transfer = null;
    $this->showResults = false;
    $this->errorMessage = '';
  }

  public function showTransferDetails()
  {
    if ($this->transfer && is_object($this->transfer) && isset($this->transfer->id)) {
      // Send only the transfer ID to the existing ShowDetails component
      $this->dispatch('showTransferDetails', $this->transfer->id);
    } else {
      Log::error('Invalid transfer object when trying to show details', [
        'transfer' => $this->transfer,
        'transfer_type' => gettype($this->transfer),
        'is_object' => is_object($this->transfer),
        'has_id' => isset($this->transfer->id) ?? false
      ]);
    }
  }

  public function render()
  {
    return view('livewire.components.transfer-search');
  }
}
