<?php

namespace App\Livewire\Components;

use App\Jobs\ImportRowJob;
use Illuminate\Support\Facades\Bus;
use Illuminate\Support\Facades\Cache;
use Livewire\Component;
use Livewire\WithFileUploads;
use Maatwebsite\Excel\Facades\Excel;

class Import extends Component
{
    use WithFileUploads;

    public $model; // Fully qualified class name (e.g., App\Models\Tenant)
    public $file;
    public $columns = [];
    public $headers = [];
    public $mappings = [];
    public $previewData = [];
    public $batchId;
    public $progress = 0;
    public $errorCount = 0;
    public function updatedFile()
    {
        $this->validate([
            'file' => 'required|file|mimes:xlsx,csv,xls',
        ]);

        $data = Excel::toArray([], $this->file);
        $this->headers = $data[0][0] ?? [];
        $this->previewData = array_slice($data[0], 1, 5);

        $this->columns = method_exists($this->model, 'importFields')
            ? array_keys($this->model::importFields())
            : [];
        $this->mappings = $this->guessMappings($this->headers, $this->columns);
    }

    public function import()
    {
        $this->validate([
            'file' => 'required|file|mimes:xlsx,csv,xls',
            'model' => 'required',
            'mappings' => 'required|array',
        ]);

        $rows = Excel::toArray([], $this->file)[0];
        $header = $rows[0];
        $rows = array_slice($rows, 1); // remove header row

        $mappedRows = [];

        foreach ($rows as $row) {
            $mappedRow = [];
            foreach ($this->mappings as $dbField => $headerName) {
                $index = array_search($headerName, $header);
                $mappedRow[$dbField] = $index !== false ? ($row[$index] ?? null) : null;
            }
            $mappedRows[] = $mappedRow;
        }


        // Create job instances for all mapped rows
        $jobs = collect($mappedRows)->map(fn($row) => new ImportRowJob($this->model, $row, $this->mappings))->toArray();

        // Dispatch a single batch with all jobs
        $batch = Bus::batch($jobs)->name('Import ' . class_basename($this->model))->dispatch();

        $this->batchId = $batch->id;

        session()->flash('success', 'Import has started. You will be notified when it finishes.');
    }

    public function getProgress()
    {
        if (!$this->batchId) return;

        $batch = Bus::findBatch($this->batchId);

        if ($batch) {
            $this->progress = $batch->progress();
        }
        $this->errorCount = Cache::get("import_errors_{$this->batchId}", 0);
    }

    protected function guessMappings(array $headers, array $fields): array
    {
        $map = [];

        foreach ($fields as $field) {
            $matched = false;
            $fieldLower = strtolower($field);

            foreach ($headers as $header) {
                $headerClean = strtolower(preg_replace('/[^a-z0-9]/i', '', $header));

                if ($headerClean === $fieldLower || str_contains($headerClean, $fieldLower)) {
                    $map[$field] = $header;
                    $matched = true;
                    break;
                }

                // Check synonyms
                if (isset($this->fieldSynonyms[$field])) {
                    foreach ($this->fieldSynonyms[$field] as $synonym) {
                        $synonymClean = strtolower(preg_replace('/[^a-z0-9]/i', '', $synonym));
                        if ($headerClean === $synonymClean || str_contains($headerClean, $synonymClean)) {
                            $map[$field] = $header;
                            $matched = true;
                            break 2;
                        }
                    }
                }
            }

            // fallback
            if (! $matched) {
                $map[$field] = null;
            }
        }

        return $map;
    }


    public function render()
    {
        return view('livewire.components.import');
    }
}
