<?php

namespace App\Livewire\Components;

use App\Models\Agent;
use App\Models\CustomLedgerAccount;
use App\Models\Currency;
use Livewire\Component;
use App\Models\User;
use Illuminate\Support\Facades\Log;

class AccountSearch extends Component
{
  public $searchAccount = '';
  public $account = null;
  public $showResults = false;
  public $errorMessage = '';

  protected $rules = [
    'searchAccount' => 'required|min:3|max:50'
  ];

  protected function messages()
  {
    return [
      'searchAccount.required' => __('Please enter an account identifier'),
      'searchAccount.min' => __('Search term must be at least 3 characters'),
      'searchAccount.max' => __('Search term must not exceed 50 characters')
    ];
  }

  public function performSearch()
  {
    // Clear previous results
    $this->errorMessage = '';
    $this->account = null;
    $this->showResults = false;

    // Simple validation - just check if search term is not empty
    if (empty($this->searchAccount)) {
      $this->errorMessage = __('Please enter an account code.');
      return;
    }

    try {
      // Search for account by code with names relation
      $account = CustomLedgerAccount::with('names')
        ->where('code', $this->searchAccount)
        ->first();

      // Always show results after search attempt
      $this->showResults = true;

      if ($account) {
        $this->account = $account;
      } else {
        $this->account = null;
        $this->errorMessage = __('No account found with the provided search term.');
      }
    } catch (\Exception $e) {
      $this->account = null;
      $this->showResults = true;
      $this->errorMessage = __('An error occurred while searching. Please try again.');
      Log::error('Account search error', ['search_term' => $this->searchAccount, 'error' => $e->getMessage()]);
    }
  }

  public function clearSearch()
  {
    $this->searchAccount = '';
    $this->account = null;
    $this->showResults = false;
    $this->errorMessage = '';
  }

  public function showAccountDetails()
  {
    if ($this->account) {
      // Dispatch event to show account details
      $this->dispatch('showAccountDetails', $this->account->id);
    } else {
      Log::error('Invalid account object when trying to show details', [
        'account' => $this->account,
        'account_type' => gettype($this->account),
        'is_object' => is_object($this->account),
        'has_id' => isset($this->account->id) ?? false
      ]);
    }
  }

  // Balance functions - using CustomLedgerAccount methods
  public function getBalances($accountCode)
  {
    // Get balances using the CustomLedgerAccount's getAgentBalances method
    $account = Agent::where('code', $accountCode)->first();
    return $account ? $account->getAgentBalances() : collect();
  }

  public function getBalanceLimit($accountCode)
  {
    // Get balance limit for CustomLedgerAccount
    $account = Agent::where('code', $accountCode)->first();
    return $account ? $account->balance_limit : 0;
  }

  public function getBaseCurrencyBalance($balances)
  {
    // Get base currency balance from the balances collection
    if ($balances && $balances->count() > 0) {
      $baseBalance = $balances->where('currency', 'USD')->first();
      return $baseBalance ? $baseBalance['balance'] : 0;
    }
    return 0;
  }

  public function getAccountName($account)
  {
    // Get account name in current language, fallback to first available name
    if ($account && $account->names && $account->names->count() > 0) {
      $currentLocale = app()->getLocale();

      // Try to get name in current language
      $nameInCurrentLang = $account->names->where('language', $currentLocale)->first();
      if ($nameInCurrentLang) {
        return $nameInCurrentLang->name;
      }

      // Fallback to first available name
      return $account->names->first()->name;
    }

    // Fallback to account name field if names relation is not available
    return $account->name ?? __('N/A');
  }

  public function getCurrencyName($currencyCode)
  {
    // Get currency name from Currency model with translation support
    $currency = Currency::where('code', $currencyCode)->first();
    if ($currency) {
      return $currency->getTranslation('name', app()->getLocale()) ?? $currencyCode;
    }
    return $currencyCode;
  }

  public function mount()
  {
    // Component mounted successfully
  }

  public function render()
  {
    return view('livewire.components.account-search');
  }
}
