<?php

namespace App\Http\Livewire;

use App\Models\BalanceOverwrite;
use Illuminate\Support\Facades\Log;
use Livewire\Component;
use Livewire\WithPagination;

class BalanceTable extends Component
{
    use WithPagination;

    public array $currencies = [];
    public bool $showRemittanceDetails = false;
    public array $filters = [];
    public int $perPage = 20;
    public array $selectedColumns = [];

    protected $listeners = ['refreshTable' => '$refresh'];

    public function mount()
    {
        $this->currencies = BalanceOverwrite::pluck('currency')->unique()->sort()->values()->toArray();
        $this->selectedColumns = $this->getDefaultVisibleColumns();
    }

    public function getDefaultVisibleColumns(): array
    {
        return array_merge(
            ['account_code', 'account_name'],
            $this->getCurrencyColumns()
        );
    }

    public function getProcessedDataProperty()
    {
        $query = BalanceOverwrite::selectRaw('
                ledgerUuid,
                currency,
                SUM(CASE WHEN balance > 0 THEN balance ELSE 0 END) as credit,
                SUM(CASE WHEN balance < 0 THEN ABS(balance) ELSE 0 END) as debit
            ')
            ->groupBy('ledgerUuid', 'currency')
            ->with('account.names');

        if (!empty($this->filters['currencies']) && is_array($this->filters['currencies'])) {
            $query->whereIn('currency', $this->filters['currencies']);
        }

        $raw = $query->get();

        $grouped = [];
        $totals = [];
        $remittanceTotals = [];
        $commissionTotals = [];

        foreach ($raw as $row) {
            $accountCode = $row->account->code ?? 'N/A';
            $accountName = $row->account->names->where("language", app()->getLocale())->first()->name ?? 'N/A';
            $currency = $row->currency;
            $credit = $row->credit;
            $debit = $row->debit;

            $key = $accountCode;

            if (!isset($grouped[$key])) {
                $grouped[$key] = [
                    'account_code' => $accountCode,
                    'account_name' => $accountName,
                ];
            }

            $grouped[$key]["{$currency}_credit"] = $credit;
            $grouped[$key]["{$currency}_debit"] = $debit;

            $totals["{$currency}_credit"] = ($totals["{$currency}_credit"] ?? 0) + $credit;
            $totals["{$currency}_debit"] = ($totals["{$currency}_debit"] ?? 0) + $debit;

            // حوالات غير مستلمة
            if (str_starts_with($accountCode, '13')) {
                $remittanceTotals["{$currency}_credit"] = ($remittanceTotals["{$currency}_credit"] ?? 0) + $credit;
                $remittanceTotals["{$currency}_debit"] = ($remittanceTotals["{$currency}_debit"] ?? 0) + $debit;

                if (!$this->showRemittanceDetails) {
                    unset($grouped[$key]);
                }
            }

            // العمولات
            if (str_starts_with($accountCode, '51')) {
                $commissionTotals["{$currency}_credit"] = ($commissionTotals["{$currency}_credit"] ?? 0) + $credit;
                $commissionTotals["{$currency}_debit"] = ($commissionTotals["{$currency}_debit"] ?? 0) + $debit;

                if (!$this->showRemittanceDetails) {
                    unset($grouped[$key]);
                }
            }
        }

        // صف المجموع الكلي
        $totalsRow = [
            'account_code' => '',
            'account_name' => 'مجموع الصناديق',
            'is_total_row' => true,
        ];
        foreach ($this->getCurrencyColumns() as $col) {
            $totalsRow[$col] = $totals[$col] ?? 0;
        }

        // صف مجموع حوالات غير مستلمة
        $remittanceRow = null;
        if (!$this->showRemittanceDetails && !empty($remittanceTotals)) {
            $remittanceRow = [
                'account_code' => '13xx',
                'account_name' => 'مجموع الحوالات غير المستلمة',
                'is_remittance_row' => true,
            ];
            foreach ($this->getCurrencyColumns() as $col) {
                $remittanceRow[$col] = $remittanceTotals[$col] ?? 0;
            }
        }

        $final = [$totalsRow];

        if ($remittanceRow) {
            $final[] = $remittanceRow;
        }

        // صف مجموع العمولات
        if (!$this->showRemittanceDetails && !empty($commissionTotals)) {
            $commissionRow = [
                'account_code' => '51xx',
                'account_name' => 'مجموع العمولات',
                'is_commission_row' => true,
            ];
            foreach ($this->getCurrencyColumns() as $col) {
                $commissionRow[$col] = $commissionTotals[$col] ?? 0;
            }

            $final[] = $commissionRow;
        }

        return array_merge($final, array_values($grouped));
    }

    public function getColumns(): array
    {
        $columns = [
            'account_code' => [
                'title' => 'كود الحساب',
                'class' => 'text-center',
                'visible' => true,
            ],
            'account_name' => [
                'title' => 'اسم الحساب',
                'class' => 'text-start',
                'visible' => true,
            ],
        ];

        $filteredCurrencies = !empty($this->filters['currencies']) && is_array($this->filters['currencies'])
            ? $this->filters['currencies']
            : $this->currencies;

        foreach ($filteredCurrencies as $currency) {
            $columns["{$currency}_credit"] = [
                'title' => "{$currency} لنا",
                'class' => 'text-danger text-end',
                'visible' => in_array("{$currency}_credit", $this->selectedColumns),
            ];

            $columns["{$currency}_debit"] = [
                'title' => "{$currency} علينا",
                'class' => 'text-success text-end',
                'visible' => in_array("{$currency}_debit", $this->selectedColumns),
            ];
        }

        return $columns;
    }

    protected function getCurrencyColumns(): array
    {
        $activeCurrencies = !empty($this->filters['currencies']) && is_array($this->filters['currencies'])
            ? $this->filters['currencies']
            : $this->currencies;

        $columns = [];
        foreach ($activeCurrencies as $currency) {
            $columns[] = "{$currency}_credit";
            $columns[] = "{$currency}_debit";
        }

        return $columns;
    }

    public function updatedFilters()
    {
        $this->resetPage();
        $this->selectedColumns = $this->getDefaultVisibleColumns();
    }

    public function toggleColumn($column)
    {
        if (in_array($column, $this->selectedColumns)) {
            $this->selectedColumns = array_diff($this->selectedColumns, [$column]);
        } else {
            $this->selectedColumns[] = $column;
        }
    }

    public function render()
    {
        $data = collect($this->processedData)->paginate($this->perPage);
        $columns = $this->getColumns();

        return view('balance-table', [
            'data' => $data,
            'columns' => $columns,
        ]);
    }
}
