<?php

namespace App\Livewire\AgentPanel\Transfer;

use Livewire\Component;
use Abivia\Ledger\Models\LedgerCurrency;
use App\Events\RefreshDataTable;
use App\Exceptions\CreditLimitExceededException;
use App\Livewire\Components\Transfer;
use App\Models\Agent;
use App\Models\ApiConfig;
use App\Models\AttachmentType;
use App\Models\Client;
use App\Models\Currency;
use App\Models\LedgerTransfer;
use App\Services\Transfer\TransferService;
use Illuminate\Support\Facades\Log;

use Illuminate\Support\Str;
use Livewire\WithFileUploads;
use Livewire\Attributes\Validate;
use App\Models\FormFieldConfig;
use App\Models\LedgerSyp;
use App\Models\Reason;
use App\Services\ExchangeRates\CurrencyConverter;
use App\Services\TransferCreditLimitValidator;
use Illuminate\Container\Attributes\Auth;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;

class AddSyrianTransfer extends Component
{
    use WithFileUploads;
    #[Validate('mimes:jpeg,png,jpg,pdf|max:1024')] // 1MB Max
    public $totalToReceive = 0;
    public $debit = 0, $credit = 0, $transferFee = 0, $total = 0, $revenue = 0;
    public $attachmentTypes = [];
    public $senderAgent, $receiverAgent;
    public $clientsSearchUrl = "/Client/Search";
    public $searchUrl = "/Client/Search";
    public  $source = "internal";
    public  $agentId = null;
    public  $external_flag = 0;
    public $send_flag = false;
    public $add_client = false;
    public $client;
    public  $reasons = [];
    public $transfer = [
        'sender_id' => null,
        'receiver_id' => null,
        'currency' => 'SYP', // default currency
        'delivery_currency' => 'SYP',
        'amount' => null,
        'sender_fee' => 0,
        'send_fee' => 0,
        'receiver_fee' => 0,
        'transfer_date' => null,
        'note' => null,
        'type' => 'transfer',
        'rate' => 1,
        'external_ref' => null,
        'delivery_amount' => 0
    ];
    public  $recipient = [
        'name' => '',
        'phone' => '',
        'sender_identity_number' => '',
    ];
    // Auto-filled sender details
    public $senderDetails = [
        'name' => null,
        'phone' => null,
        'address' => null,
        'balance' => null,
    ];

    // Auto-filled receiver details
    public $receiverDetails = [
        'name' => null,
        'phone' => null,
        'address' => null,
    ];
    public $formFieldConfigs = [];
    // Available currencies and agents
    public $currencies = [];
    public $senders = [];
    public $receivers = [];
    public $address = null;
    public  $attatchments = [];
    public  $attachment_types = [];
    public  $attachment_type = "attachments";
    protected $rules = [
        'transfer.sender_id' => 'required|exists:agents,id',
        'transfer.receiver_id' => 'required| different:transfer.sender_id',
        'transfer.currency' => 'required|exists:ledger_currencies,code',
        'transfer.amount' => 'required|numeric|min:1',
        'transfer.send_fee' => 'required|numeric|min:0',
        //    'transfer.delivery_amount' => 'required|numeric|min:1',
        'transfer.receiver_fee' => 'required|numeric|min:0',
        'transfer.transfer_date' => 'nullable|date',
        'recipient.name' => 'required',
        'recipient.phone' => 'required',
    ];


    protected function  getListeners()
    {
        return [
            'getClientData' => 'getClientData',
            'refreshAttachmentTypes' => 'refreshAttachmentTypes',
            'editTransfer' => 'editTransfer',
            'newTransfer' => 'newTransfer',
            "AddNewClientEvent" => "AddNewClientEvent",
            "CancelAddClientEvent" => "CancelAddClientEvent",
            "SendTransfer" => "SendTransfer",
        ];
    }
    public  function  AddNewClientEvent()
    {
        $this->add_client = true;
    }
    public function  CancelAddClientEvent()
    {
        $this->add_client = false;
    }
    public function newTransfer()
    {
        $this->recipient = [
            'name' => '',
            'phone' => '',
            'sender_identity_number' => '',
        ];
        $this->transfer = [
            'sender_id' => null,
            'receiver_id' => null,
            'currency' => 'SYP', // default currency
            'delivery_currency' => 'SYP',
            'amount' => 0,
            'sender_fee' => 0,
            'send_fee' => 0,
            'receiver_fee' => 0,
            'transfer_date' => null,
            'note' => null,
            'type' => 'transfer',
            'rate' => 1,
            'external_ref' => null,
            'reason_id' => 1
        ];
    }


    public function editTransfer($id)
    {

        $transfer = LedgerSyp::where("id", $id)->first();
        $receiver = $transfer->receiver;
        $sender = $transfer->sender;
        $this->senderDetails = [
            'name' => $sender?->name,
            'phone' => $sender?->phone,
            'address' => $sender?->address?->full_address,
        ];
        $this->receiverDetails = [
            'name' => $receiver?->name,
            'phone' => $receiver?->phone,
            'address' => $receiver?->address?->full_address,
        ];
        $this->receiverAgent = $receiver;
        $this->transfer = $transfer->toArray();
        if ($transfer->isExternalTransfer()) {
            $this->source = "external";
            $this->external_flag = 1;
            $this->transfer['receiver_id'] = $this->transfer["api_id"] . "_" . $this->transfer["external_agent_id"];
        } else {
            $this->external_flag = 0;
            $this->source = "internal";
            $this->transfer['receiver_id'] = "internal_" . $this->transfer["receiver_id"];
        }
        $this->recipient = $transfer->customer_details;
        $this->recipient['sender_identity_number'] = $this->recipient['identity_number'];
        $this->searchSender();
    }
    public function  refreshAttachmentTypes()
    {
        $this->attachmentTypes = AttachmentType::all();
    }
    public function  UploadAttatchments($transfer_id)
    {
        $transfer = LedgerSyp::find($transfer_id);
        $attachment_type = $this->attachment_type;
        if (count($this->attatchments) > 0) {
            $transfer->clearMediaCollection($attachment_type ?? 'attatchments');
            foreach ($this->attatchments as $attatchment) {

                $transfer
                    ->addMedia($attatchment->getRealPath())
                    ->usingFileName($attatchment->getClientOriginalName())
                    ->toMediaCollection($attachment_type ?? 'attatchments');
            }
        }
    }
    public function  searchSender()
    {
        $recipent = $this->recipient;

        $this->validate([
            'recipient.sender_identity_number' => 'required',
        ]);

        $client = Client::where('phone', '=', $recipent['sender_identity_number'])->orWhere('identity_number', '=', $recipent['sender_identity_number'])->orWhere('name', '=', $recipent['sender_identity_number'])->orWhere('id', '=', $recipent['sender_identity_number'])->first();

        if ($client) {
            $this->recipient = $client->toArray();
            $this->recipient['sender_client_id'] = $client->id;
            //  $this->recipient['sender_identity_number'] = $client->identity_number;
            $this->transfer['client_id'] = $client->id;
            $this->transfer['client_sender_id_number'] =  $client->identity_number;

            $this->dispatch("LoadClientData", ['id' => $client->id]);
        } else {
            $this->dispatch("sweetalert:error", ['msg' => "No Client Found", "title" => "Error", "type" => "error"]);
        }
    }
    public  function  getClientData($id)
    {
        $id = $id['id'];
        $client = Client::find($id);
        $this->client = $client;
        $this->recipient['recipient_name'] = $client->name;
        $this->recipient['recipient_phone'] = $client->phone;
        $this->recipient['sender_client_identity_number'] = $client->identity_number;
        $this->recipient['sender_client_id'] = $client->id ?? null;
        $this->recipient['sender_identity_number'] = $client->id ?? null;


        $this->add_client = false;
    }
    public function mount($transfer_id = null)
    {
        $this->transfer['sender_id'] = auth()->guard('agent')->user()->agent->id ?? 1;
        $this->senderAgent = auth()->guard('agent')->user()->agent;

        $this->currencies = Currency::get();
        $this->transfer['currency'] = "SYP";
        $this->senders = LedgerSyp::GetSourceAgents();
        $this->receivers = LedgerSyp::GetDestinationAgents();
        $this->reasons = Reason::get();
        $this->transfer['transfer_date'] = now()->format('Y-m-d\TH:i');
        $this->attachmentTypes = AttachmentType::get();
        $this->formFieldConfigs = FormFieldConfig::forForm('transfer_form');
        if ($transfer_id) {
            $this->editTransfer($transfer_id);
        }
    }
    public function Updated($key, $value)
    {
        $this->validateOnly($key);
        if (in_array($key, ['transfer.currency', 'transfer.amount', 'transfer.sender_id'])) {
            $this->CheckBalance();
        }
        if ($key != "transfer.rate")
            $this->Calculate();
    }
    public function Calculate()
    {


        $this->transfer['delivery_amount'] = $this->transfer['amount'];
    }
    public  function UpdatedTransferRate($value)
    {

        $converter = new CurrencyConverter();
        $from_amount = $this->transfer['amount'] ?? 0;
        $from_currency = $this->transfer['currency'] ?? "EUR";
        $to_currency = $this->transfer['delivery_currency'] ?? "EUR";
        $factor = getCurrencyFactor($from_currency, $to_currency, 'transfer');

        $this->transfer['rate_factor'] = $factor;
        $converted = $converter->convert(
            $from_amount,
            $from_currency,
            $to_currency,
            $this->currencies,
            'transfer',
            $value,
            $factor == "multiply" ? "*" : "/"
        );

        $this->transfer['delivery_amount'] = $converted;
    }
    public  function UpdatedTransferRateFactor($value)
    {

        $converter = new CurrencyConverter();
        $from_amount = $this->transfer['amount'] ?? 0;
        $from_currency = $this->transfer['currency'] ?? "EUR";
        $to_currency = $this->transfer['delivery_currency'] ?? "EUR";
        $factor = $value; // getCurrencyFactor($from_currency, $to_currency, 'transfer');
        $rate = $this->transfer['rate'];
        $this->transfer['rate_factor'] = $factor;
        $converted = $converter->convert(
            $from_amount,
            $from_currency,
            $to_currency,
            $this->currencies,
            'transfer',
            $rate,
            $factor == "multiply" ? "*" : "/"
        );

        $this->transfer['delivery_amount'] = $converted;
    }
    public function UpdatedTransferDeliveryCurrency($value)
    {
        // Update delivery currency when transfer currency changes
        $this->transfer['delivery_currency'] = $value;
        $this->transfer['rate'] = Currency::getExchangeRate($value);
        $this->transfer['rate_factor'] = Currency::getRateFactor($value);
        $this->Calculate();
    }

    public function UpdatedTransferAmount($value)
    {
        $this->Calculate();

        // $this->transfer['delivery_amount'] = $value * $this->transfer['exchange_rate'];
    }

    public function updatedTransferSenderId($value)
    {

        if ($value) {
            $sender = Agent::with('address')->find($value);
            $this->senderAgent = $sender;
            $this->senderDetails = [
                'name' => $sender?->name,
                'phone' => $sender?->phone,
                'address' => $sender?->address?->full_address,
                'balance' => 0, // $sender->getBalance($this->transfer['currency']),
            ];
        } else {
            $this->resetSenderDetails();
        }
    }
    public function updatedTransfer($value, $field)
    {
        $this->getFee();
        // $this->getTotalToReceive();
        // Recalculate when transfer data changes
        if (in_array($field, ['amount', 'receiver_fee'])) {
            //    unset($this->totalToReceive);
        }
        //  $this->calclulateDabitandCredit();
    }
    public function updatedTransferReceiverId($value)
    {
        if ($value) {
            $source = "internal";
            $agentId = $value;
            // [$source, $agentId] = explode('_', $value);
            $this->source = $source;
            $this->agentId = $agentId;
            if ($source == "internal") {
                $receiver = Agent::with('address')->find($agentId);
                $this->receiverDetails = [
                    'name' => $receiver->name,
                    'phone' => $receiver->phone,
                    'address' => $receiver->address?->full_address,
                ];
                $this->receiverAgent = $receiver;
                $this->address = $receiver->getFullAddress();
            } else {
                $config = ApiConfig::find($source);
                $api = ApiConnection($config);
                $data = $api->getClientData($agentId);
                $this->address = $data['address'] ?? "";
            }
        }
    }



    // When currency changes
    public function updatedTransferCurrency()
    {
        $this->transfer['delivery_currency'] = $this->transfer['currency'];
    }

    public function getTotalToReceive()
    {
        $amount = $this->transfer['amount'] ?? 0;
        $receiverFee = $this->transfer['receiver_fee'] ?? 0;
        //$total = max(0, $amount - $receiverFee);
        $total = $amount;
        $this->totalToReceive = $total;
    }

    // Calculate total amount to be deducted from sender
    public function getTotalToSendProperty()
    {
        $amount = $this->transfer['amount'] ?? 0;
        $senderFee = $this->transfer['sender_fee'] ?? 0;
        return $amount + $senderFee;
    }

    // Reset sender details
    protected function resetSenderDetails()
    {
        $this->senderDetails = [
            'name' => null,
            'phone' => null,
            'address' => null,
            'balance' => null,
        ];
    }

    // Reset receiver details
    protected function resetReceiverDetails()
    {
        $this->receiverDetails = [
            'name' => null,
            'phone' => null,
            'address' => null,
        ];
    }
    public function calclulateDabitandCredit()
    {
        $amount = $this->transfer['amount'] ?? 0;
        $receiverFee = $this->transfer['receiver_fee'] ?? 0;
        $senderFee = $this->transfer['sender_fee'] ?? 0;
        $debit = $amount + $senderFee;
        $credit = $amount + $receiverFee;
        $revenue = $senderFee - $receiverFee;
        $this->debit = $debit;
        $this->credit = $credit;
        $this->revenue = $revenue;
    }
    public  function  SendTransfer()
    {
        $this->transfer['secret'] = rand(1000, 9999);
        $this->transfer['reference'] = $this->generateRandomTransferNumber();

        $lockKey = 'lock:transfer:' . $this->transfer['reference'];
        $lock = Cache::lock($lockKey, 10); // Lock for 10 seconds
        try {


            if (isset($this->transfer['id'])) {
                $this->UpdateTransfer();
                return;
            }


            if (!$lock->get()) {
                $this->dispatch("sweetalert:error", ['msg' => __("alerts.transfer_locked"), "title" => "Error", "type" => "error"]);
                return;
            }


            $creditlimitValidator = new TransferCreditLimitValidator();

            DB::beginTransaction();

            // $canSend = $this->senderAgent->canSendIn($this->transfer['currency']);
            // if (!$canSend) {
            //     $this->dispatch("sweetalert:error", ['msg' => __("alerts.sender_currency_not_allowed"), "title" => "Error", "type" => "error"]);
            //     DB::rollBack();
            //     return;
            // }

            // if ($this->source == "internal") {
            //     $canReceive = $this->receiverAgent->canReceiveIn($this->transfer['delivery_currency']);
            //     $this->transfer['receiver_id'] = $this->receiverAgent->id;
            //     if (!$canReceive) {
            //         $this->dispatch("sweetalert:error", ['msg' => __("alerts.receiver_currency_not_allowed"), "title" => "Error", "type" => "error"]);
            //         DB::rollBack();
            //         return;
            //     }
            // }

            $creditlimitValidator->validate($this->senderAgent, $this->transfer['currency'], $this->transfer['amount']);

            if ($this->source != "internal") {
                $config = ApiConfig::with("apiType")->find((int)$this->source);
                $this->transfer['receiver_id'] = $config->agent_id;
                $receiver_id = $this->agentId;

                $result = sendExternalTransaction($config, $this->transfer, $receiver_id);

                if (!$result['success']) {
                    $this->dispatch("sweetalert:error", ['msg' => $result['message'], "title" => "Error", "type" => "error"]);
                    DB::rollBack();
                    return;
                }
                $this->transfer['api_id'] = $config->id;
                $this->transfer['external_agent_id'] = $receiver_id;
                $this->transfer['external_ref'] = $result['ref'];
                $this->transfer['secret'] = $result['password'] !== "" ? $result['password'] : rand(1000, 9999);
                $this->transfer['receiver_fee'] = $result['tax'] ?? 0;
            }

            $this->transfer['customer_details'] = $this->recipient;

            $service = new TransferService();
            $transfer = $service->create($this->transfer);

            DB::commit();

            $this->dispatch('saveMedia', \App\Models\LedgerTransfer::class, $transfer->id);
            $address = [
                'full_address' => $this->address
            ];
            $transfer->createOrUpdateAddress($address);

            event(new RefreshDataTable("transfer-table"));
            event(new RefreshDataTable("livewatch-table"));

            $this->dispatch("sweetalert:success", ['msg' => "Transfers Added Successfully", "title" => "Success", "type" => "success"]);

            $this->resetForm();

            $this->dispatch('ShowTransferReceipt', ['transfer_id' => $transfer->id]);
        } catch (ValidationException $e) {
            $firstField = array_key_first($e->validator->errors()->messages());
            $firstErrorMessage = $e->validator->errors()->first($firstField);
            $this->dispatch('focus-error', ['field' => $firstField]);
            $this->dispatch("sweetalert:error", [
                'msg' => $firstErrorMessage,
                'title' => __("alerts.error"),
                'type' => "error"
            ]);
            DB::rollBack();
            throw $e; // rethrow so Livewire still shows errors
        } catch (CreditLimitExceededException $e) {
            Log::error($e);
            DB::rollBack();
            $this->dispatch("sweetalert:error", ['msg' => $e->getMessage(), "title" => "Error", "type" => "error"]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e);
            $this->dispatch("sweetalert:error", ['msg' => $e->getMessage(), "title" => "Error", "type" => "error"]);
        } finally {
            Log::info('Transfer added', ['transfer' => $this->transfer]);
            optional($lock)->release(); // Always release the lock
        }

        $this->dispatch("UnBlockUI");
    }
    public function confirmTransfer()
    {

        if (!isset($this->transfer['client_id'])) {
            $this->dispatch("sweetalert:error", ['msg' => __("alerts.client_id_required"), "title" => "Error", "type" => "error"]);
            return;
        }


        $rules = [
            'transfer.sender_id' => 'required|exists:agents,id',
            'transfer.receiver_id' => 'required|different:transfer.sender_id',
            'transfer.currency' => 'required|exists:ledger_currencies,code',
            'transfer.amount' => 'required|numeric|min:1',
            'transfer.send_fee' => 'required|numeric|min:0',
            'transfer.receiver_fee' => 'required|numeric|min:0',
            'transfer.transfer_date' => 'nullable|date',
            'recipient.name' => 'required',
            'recipient.phone' => 'nullable',
        ];

        try {
            $this->validate($rules);

            if (isset($this->transfer['id'])) {
                $this->UpdateTransfer();
                return;
            }




            $creditlimitValidator = new TransferCreditLimitValidator();



            $creditlimitValidator->validate($this->senderAgent, $this->transfer['currency'], $this->transfer['amount']);
            $currency_word = $this->getAmountInWords($this->transfer['amount'], $this->transfer['currency']);
            $this->dispatch("areyousure", ['msg' => __("alerts.areyousure"), 'emitName' => 'SendTransfer', 'action' => __("YES"), 'title' => __("alerts.areyousure_send_transfer", ["currency_word" => $currency_word, 'name' => $this->transfer['recipient_name']]), 'data' => []]);
        } catch (ValidationException $e) {
            $firstField = array_key_first($e->validator->errors()->messages());
            $firstErrorMessage = $e->validator->errors()->first($firstField);
            $this->dispatch('focus-error', ['field' => $firstField]);
            $this->dispatch("sweetalert:error", [
                'msg' => $firstErrorMessage,
                'title' => __("alerts.error"),
                'type' => "error"
            ]);
            DB::rollBack();
            throw $e; // rethrow so Livewire still shows errors
        } catch (CreditLimitExceededException $e) {


            $this->dispatch("sweetalert:error", ['msg' => $e->getMessage(), "title" => "Error", "type" => "error"]);
        } catch (\Exception $e) {

            $this->dispatch("sweetalert:error", ['msg' => $e->getMessage(), "title" => "Error", "type" => "error"]);
        }
    }
    public  function UpdateTransfer()
    {
        try {
            LedgerSyp::enableAuditing();
            $transfer = LedgerSyp::find($this->transfer['id']);
            $this->transfer['receiver_id'] = $this->receiverAgent->id;
            $transfer->update($this->transfer);

            if (isset($transfer['external_ref'])) {
                $receiver = $transfer->receiver;
                $config = ApiConfig::with("apiType")->where("agent_id", $receiver->id)->first();
                $class = $config->apiType->class_name;
                $service_api = new $class();
                $configArray = json_decode($config->config_json, true);
                $configArray['base_url'] = $config->base_url;
                $service_api->setConfig($configArray);
                $res = $service_api->UpdateTransaction($transfer);
            } else {
                $service = new TransferService();
                $res = $service->UpdateLedger($transfer);
            }
            if ($res['success']) {
                $this->dispatch("sweetalert:success", ['msg' => $res['message'], "title" => "Success", "type" => "success"]);
            } else {
                $this->dispatch("sweetalert:error", ['msg' => $res['message'], "title" => "Error", "type" => "error"]);
            }
        } catch (\Exception $e) {
            Log::info($e);
            Log::error('Error updating transfer', ['error' => $e->getMessage()]);
        }
    }

    public function  getFee()
    {

        $sender = Agent::find($this->transfer['sender_id']);
        if ($sender) {
            $package = $sender->CurrencyTransferPackages($this->transfer['currency']);
            if ($package && $this->transfer['amount'] != 0 && isset($this->transfer['amount'])) {
                $fees = getFeePriceFromPackage($package->id, $this->transfer['amount']);

                if (!$fees['status']) {
                    $this->addError('fee', $fees['msg']);

                    $this->send_flag = false;
                    return;
                }
                $this->send_flag = true;
                $this->transfer['send_fee'] = $fees['send_fee'];
                $this->transfer['receiver_fee'] = $fees['receive_fee'];
            }
        }
    }
    public function CheckBalance()
    {
        try {
            $currency = $this->transfer['currency'] ?? "";
            $amount = $this->transfer['amount'];
            $send_fee = $this->transfer['send_fee'] ?? 0;
            $agent = $this->senderAgent;

            if ($agent) {
                $creditlimitValidator = new TransferCreditLimitValidator();
                $creditlimitValidator->validate($this->senderAgent, $currency, $amount + $send_fee);
                $this->send_flag = true;
                Log::debug("CheckBalance: No exception, send_flag set to true");
            } else {
                $this->addError('balance', "Agent not found");
                $this->send_flag = false;
                Log::debug("CheckBalance: Agent not found, send_flag set to false");
            }
        } catch (CreditLimitExceededException $e) {
            Log::debug("CheckBalance: Credit limit exception caught, setting send_flag to false");
            Log::error($e);
            $this->addError('balance', $e->getMessage());
            $this->send_flag = false;
        } catch (\Exception $e) {
            Log::debug("CheckBalance: General exception caught, setting send_flag to false");
            Log::error($e);
            $this->send_flag = false;
            $this->addError('balance', $e->getMessage());
        }

        Log::debug("CheckBalance END: send_flag is " . var_export($this->send_flag, true));
    }

    // Reset the form
    public function resetForm()
    {
        $this->reset('transfer');
        $this->resetSenderDetails();
        $this->resetReceiverDetails();
        $this->transfer['currency'] = 'SYP';
        $this->transfer['sender_id'] = auth()->guard('agent')->user()->agent->id;
        $this->senderAgent = auth()->guard('agent')->user()->agent;
        $this->transfer['transfer_date'] = now()->format('Y-m-d\TH:i');
        $this->recipient = [
            'name' => '',
            'phone' => '',
            'sender_identity_number' => '',
        ];
        $this->resetErrorBag();
    }
    public function getAmountInWordsProperty()
    {
        if ($this->transfer['amount'] && $this->transfer['currency']) {

            return toCurrencyWords(
                floatval($this->transfer['amount']),
                $this->transfer['currency'],
                app()->getLocale()
            );
        }
        return '';
    }
    public function getAmountInWords($amount, $currency)
    {

        if ($amount && $currency) {
            return toCurrencyWords(
                floatval($amount),
                $currency,
                app()->getLocale()
            );
        }
        return '';
    }
    public function  getBalances($agent_id)
    {
        if (!$agent_id) {
            return [];
        }
        $agent = Agent::find($agent_id);
        return $agent->getAgentBalances();
    }

    public function fieldIsVisible($key)
    {
        return $this->formFieldConfigs[$key]['visible'] ?? true;
    }

    public function fieldIsRequired($key)
    {
        return $this->formFieldConfigs[$key]['required'] ?? false;
    }

    public function fieldIsLocked($key)
    {
        return $this->formFieldConfigs[$key]['always_required'] ?? false;
    }
    public  function getFieldLabel($key)
    {
        return $this->formFieldConfigs[$key]['label'] ?? Str::title($key);
    }
    public  function getFieldTabIndex($key)
    {
        return $this->formFieldConfigs[$key]['tab_index'] ?? 1;
    }
    public function UpdatedRecipientSenderIdentityNumber()
    {
        $this->searchSender();
    }
    public function generateRandomTransferNumber(): string
    {
        $branchCode = str_pad(mt_rand(0, 999), 3, '0', STR_PAD_LEFT);
        $transactionId = str_pad(mt_rand(0, 999999), 6, '0', STR_PAD_LEFT);

        return "TRF-{$branchCode}-{$transactionId}";
    }

    public function render()
    {
        return view('livewire.agent-panel.transfer.add-syrian-transfer');
    }
}
