<?php

namespace App\Livewire\AgentPanel\Transfer;

use Livewire\Component;
use Abivia\Ledger\Models\LedgerCurrency;
use App\Events\RefreshDataTable;
use App\Exceptions\CreditLimitExceededException;
use App\Exceptions\InvalidExchangeRateException;
use App\Livewire\Components\Transfer;
use App\Models\Agent;
use App\Models\ApiConfig;
use App\Models\AttachmentType;
use App\Models\Client;
use App\Models\Currency;
use App\Models\LedgerTransfer;
use App\Services\Transfer\TransferService;
use Illuminate\Support\Facades\Log;

use Illuminate\Support\Str;
use Livewire\WithFileUploads;
use Livewire\Attributes\Validate;
use App\Models\FormFieldConfig;
use App\Models\Reason;
use App\Services\ExchangeRates\CurrencyConverter;
use App\Services\TransferCreditLimitValidator;
use Illuminate\Container\Attributes\Auth;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Rawilk\Settings\Facades\Settings;
use App\Helpers\TransferValidationHelper;

class Add extends Component
{
  use WithFileUploads;
  #[Validate('mimes:jpeg,png,jpg,pdf|max:1024')] // 1MB Max
  public $totalToReceive = 0;
  public $debit = 0, $credit = 0, $transferFee = 0, $total = 0, $revenue = 0;
  public $attachmentTypes = [];
  public $senderAgent, $receiverAgent;
  public $clientsSearchUrl = "/Client/Search";
  public $searchUrl = "/Client/Search";
  public $client;
  public  $source = "internal";
  public  $agentId = null;
  public  $external_flag = 0;
  public  $send_flag = false;
  public  $suppressCalculate = false;
  public  $reasons = [];
  public  $add_client = false;
  public $transfer = [
    'sender_id' => null,
    'receiver_id' => null,
    'currency' => null, // default currency
    'delivery_currency' => null,
    'amount' => null,
    'sender_fee' => 0,
    'send_fee' => 0,
    'receiver_fee' => 0,
    'transfer_date' => null,
    'note' => null,
    'type' => 'transfer',
    'rate' => 1,
    'external_ref' => null,
    'delivery_amount' => null,
    "reason_id" => 1,
    "fee_type" => "collected"
  ];
  public  $recipient = [
    'name' => '',
    'phone' => '',
    'sender_identity_number' => '',
  ];
  // Auto-filled sender details
  public $senderDetails = [
    'name' => null,
    'phone' => null,
    'address' => null,
    'balance' => null,
  ];

  // Auto-filled receiver details
  public $receiverDetails = [
    'name' => null,
    'phone' => null,
    'address' => null,
  ];
  public $formFieldConfigs = [];
  // Available currencies and agents
  public $currencies = [];
  public  $receiver_currencies = [];
  public $senders = [];
  public $receivers = [];
  public $address = null;
  public  $attatchments = [];
  public  $attachment_types = [];
  public  $attachment_type = "attachments";
  public $is_advanced_transfer = 0;

  protected $rules = [
    'transfer.sender_id' => 'required|exists:agents,id',
    'transfer.receiver_id' => 'required| different:transfer.sender_id',
    'transfer.currency' => 'required|exists:ledger_currencies,code',
    'transfer.amount' => 'required|numeric|min:1',
    'transfer.send_fee' => 'required|numeric|min:0',
    //    'transfer.delivery_amount' => 'required|numeric|min:1',
    'transfer.receiver_fee' => 'required|numeric|min:0',
    'transfer.transfer_date' => 'nullable|date',
    'recipient.name' => 'required',
    'recipient.phone' => 'required',
  ];


  protected function  getListeners()
  {
    return [
      'getClientData' => 'getClientData',
      'refreshAttachmentTypes' => 'refreshAttachmentTypes',
      'editTransfer' => 'editTransfer',
      'newTransfer' => 'newTransfer',
      'AddNewClientEvent' => 'AddNewClientEvent',
      "CancelAddClientEvent" => "CancelAddClientEvent",
      'SendTransfer' => 'SendTransfer',

    ];
  }
  public function  CancelAddClientEvent()
  {
    $this->add_client = false;
  }
  public  function  AddNewClientEvent()
  {
    $this->add_client = true;
  }
  public function newTransfer()
  {
    $this->recipient = [
      'name' => '',
      'phone' => '',
      'sender_identity_number' => '',
    ];
    $this->transfer = [
      'sender_id' => null,
      'receiver_id' => null,
      'currency' => null, // default currency
      'delivery_currency' => null,
      'amount' => null,
      'sender_fee' => null,
      'send_fee' => null,
      'receiver_fee' => null,
      'transfer_date' => null,
      'note' => null,
      'type' => 'transfer',
      'rate' => 1,
      'external_ref' => null,
      'reason_id' => 1
    ];
  }


  public function editTransfer($id)
  {

    $transfer = LedgerTransfer::where("id", $id)->first();
    $receiver = $transfer->receiver;
    $sender = $transfer->sender;
    $this->senderDetails = [
      'name' => $sender?->name,
      'phone' => $sender?->phone,
      'address' => $sender?->address?->full_address,
    ];
    $this->receiverDetails = [
      'name' => $receiver?->name,
      'phone' => $receiver?->phone,
      'address' => $receiver?->address?->full_address,
    ];
    $this->receiverAgent = $receiver;
    $this->transfer = $transfer->toArray();
    if ($transfer->isExternalTransfer()) {
      $this->source = "external";
      $this->external_flag = 1;
      $this->transfer['receiver_id'] = $this->transfer["api_id"] . "_" . $this->transfer["external_agent_id"];
    } else {
      $this->external_flag = 0;
      $this->source = "internal";
      $this->transfer['receiver_id'] = "internal_" . $this->transfer["receiver_id"];
    }
    $this->recipient = $transfer->customer_details;
    $this->recipient['sender_identity_number'] = $this->recipient['identity_number'];
    $this->searchSender();
  }
  public function  refreshAttachmentTypes()
  {
    $this->attachmentTypes = AttachmentType::all();
  }
  public function  UploadAttatchments($transfer_id)
  {
    $transfer = LedgerTransfer::find($transfer_id);
    $attachment_type = $this->attachment_type;
    if (count($this->attatchments) > 0) {
      $transfer->clearMediaCollection($attachment_type ?? 'attatchments');
      foreach ($this->attatchments as $attatchment) {

        $transfer
          ->addMedia($attatchment->getRealPath())
          ->usingFileName($attatchment->getClientOriginalName())
          ->toMediaCollection($attachment_type ?? 'attatchments');
      }
    }
  }
  public function  searchSender()
  {
    $recipent = $this->recipient;

    $this->validate([
      'recipient.sender_identity_number' => 'required',
    ]);


    $client = Client::where('phone', '=', $recipent['sender_identity_number'])->orWhere('identity_number', '=', $recipent['sender_identity_number'])->orWhere('name', '=', $recipent['sender_identity_number'])->orWhere('id', '=', $recipent['sender_identity_number'])->first();
    if ($client) {
      $this->recipient = $client->toArray();
      $this->recipient['sender_client_id'] = $client->id;
      //  $this->recipient['sender_identity_number'] = $client->identity_number;
      $this->transfer['client_id'] = $client->id;
      $this->transfer['client_sender_id_number'] =  $client->identity_number;

      $this->dispatch("LoadClientData", ['id' => $client->id]);
    } else {
      $this->dispatch("sweetalert:error", ['msg' => "No Client Found", "title" => "Error", "type" => "error"]);
    }
  }
  public  function  getClientData($id)
  {

    $id = $id['id'];
    $client = Client::find($id);
    $this->client = $client;
    $this->recipient['recipient_name'] = $client->name;
    $this->recipient['recipient_phone'] = $client->phone;
    $this->recipient['sender_client_identity_number'] = $client->identity_number;
    $this->recipient['sender_client_id'] = $client->id ?? null;
    $this->recipient['sender_identity_number'] = $client->id ?? null;


    $this->add_client = false;
  }
  public function mount($transfer_id = null)
  {
    $this->is_advanced_transfer = Settings::get('agent.advanced_transfer_status', 0);
    $this->transfer['sender_id'] = auth()->guard('agent')->user()->agent->rootAgent()->id;
    $this->senderAgent = auth()->guard('agent')->user()->agent->rootAgent();

    $this->currencies = $this->senderAgent->getCanSendCurrency();

    // $this->senders = LedgerTransfer::GetSourceAgents();
    //  $this->receivers = LedgerTransfer::GetDestinationAgents();
    $this->reasons = Reason::get();
    $this->transfer['transfer_date'] = now()->format('Y-m-d\TH:i');
    $this->attachmentTypes = AttachmentType::get();
    $this->formFieldConfigs = FormFieldConfig::forForm('transfer_form');
    if ($transfer_id) {
      $this->editTransfer($transfer_id);
    }
  }
  public function Updated($key, $value)
  {
    $this->validateOnly($key);
    if (in_array($key, ['transfer.currency', 'transfer.amount', 'transfer.sender_id'])) {
      $this->CheckBalance();
    }
    if ($key != "transfer.rate" && $key != "transfer.delivery_amount")
      $this->Calculate();
  }
  public function UpdatedTransferDeliveryAmount($value)
  {
    try {
      if ($this->suppressCalculate) {
        return;
      }

      $converter = new CurrencyConverter();
      $delivery_amount = $value ?? 0;
      $from_currency = $this->transfer['currency'] ?? "EUR";
      $to_currency = $this->transfer['delivery_currency'] ?? "EUR";

      if (!$delivery_amount || !$to_currency) {
        return;
      }

      $factor = getCurrencyFactor($from_currency, $to_currency, 'transfer');
      $rate = exchange_rate($from_currency, $to_currency, 'transfer');
      if (!$rate || $rate == null) {
        $this->dispatch("sweetalert:error", ['msg' => "Exchange rate not found", "title" => "Error", "type" => "error"]);
        $this->send_flag = false;
        return;
      }
      // Calculate in reverse (delivery_amount → amount)
      $this->suppressCalculate = true;
      $this->transfer['rate'] = $rate ?? $this->transfer['rate'];
      $this->transfer['rate_factor'] = $factor;
      if ($factor === 'divide') {
        $this->transfer['amount'] = $delivery_amount * $rate;
      } else {
        $this->transfer['amount'] =  $delivery_amount / $rate;
      }

      $this->suppressCalculate = false;
    } catch (InvalidExchangeRateException $e) {
      $this->addError('transfer.rate', $e->getMessage());
      $this->send_flag = false;
    }
  }
  public function Calculate()
  {
    try {
      if ($this->suppressCalculate) {
        return;
      }

      $fee_type = $this->transfer['fee_type'] ?? "collected";
      if ($fee_type == "included") {
        $this->transfer['amount'] = $this->transfer['amount'] - $this->transfer['send_fee'];
      }
      $converter = new CurrencyConverter();
      $from_amount = $this->transfer['amount'] ?? 0;
      $from_currency = $this->transfer['currency'] ?? "EUR";
      $to_currency = $this->transfer['delivery_currency'] ?? "EUR";

      if (!$from_amount || !$from_currency) {
        return;
      }

      $factor = getCurrencyFactor($from_currency, $to_currency, 'transfer');
      $this->transfer['rate_factor'] = $factor;

      $rate = exchange_rate($from_currency, $to_currency, 'transfer');
      $this->transfer['rate'] = $this->transfer['rate'] ?? 1;
      if (!$rate || $rate == null) {
        $this->dispatch("sweetalert:error", ['msg' => "Exchange rate not found", "title" => "Error", "type" => "error"]);
        $this->send_flag = false;
        return;
      }
      $converted = $converter->convert(
        $from_amount,
        $from_currency,
        $to_currency,
        $this->currencies,
        'transfer',
        $rate,
        $factor
      );

      $this->suppressCalculate = true;
      $this->transfer['rate'] = $rate ?? $this->transfer['rate'];
      $this->transfer['delivery_amount'] = $converted;
      $this->suppressCalculate = false;
    } catch (InvalidExchangeRateException $e) {
      $this->addError('transfer.rate', $e->getMessage());
      $this->send_flag = false;
    }
  }
  public  function UpdatedTransferRate($value)
  {

    $converter = new CurrencyConverter();
    $from_amount = $this->transfer['amount'] ?? 0;
    $from_currency = $this->transfer['currency'] ?? "EUR";
    $to_currency = $this->transfer['delivery_currency'] ?? "EUR";
    $factor = getCurrencyFactor($from_currency, $to_currency, 'transfer');

    $this->transfer['rate_factor'] = $factor;
    $converted = $converter->convert(
      $from_amount,
      $from_currency,
      $to_currency,
      $this->currencies,
      'transfer',
      $value,
      $factor == "multiply" ? "*" : "/"
    );

    $this->transfer['delivery_amount'] = $converted;
  }
  public  function UpdatedTransferRateFactor($value)
  {

    $converter = new CurrencyConverter();
    $from_amount = $this->transfer['amount'] ?? 0;
    $from_currency = $this->transfer['currency'] ?? "EUR";
    $to_currency = $this->transfer['delivery_currency'] ?? "EUR";
    $factor = $value; // getCurrencyFactor($from_currency, $to_currency, 'transfer');
    $rate = $this->transfer['rate'];
    $this->transfer['rate_factor'] = $factor;
    $converted = $converter->convert(
      $from_amount,
      $from_currency,
      $to_currency,
      $this->currencies,
      'transfer',
      $rate,
      $factor == "multiply" ? "*" : "/"
    );

    $this->transfer['delivery_amount'] = $converted;
  }
  public function UpdatedTransferDeliveryCurrency($value)
  {
    // Update delivery currency when transfer currency changes
    $this->transfer['delivery_currency'] = $value;
    $this->transfer['rate'] = Currency::getExchangeRate($value);
    $this->transfer['rate_factor'] = Currency::getRateFactor($value);
    $this->Calculate();
  }

  public function UpdatedTransferAmount($value)
  {
    $this->Calculate();

    // $this->transfer['delivery_amount'] = $value * $this->transfer['exchange_rate'];
  }

  public function updatedTransferSenderId($value)
  {

    if ($value) {
      $sender = Agent::with('address')->find($value);
      $this->senderAgent = $sender;
      $this->senderDetails = [
        'name' => $sender?->name,
        'phone' => $sender?->phone,
        'address' => $sender?->address?->full_address,
        'balance' => 0, // $sender->getBalance($this->transfer['currency']),
      ];
    } else {
      $this->resetSenderDetails();
    }
  }
  public function updatedTransfer($value, $field)
  {
    $this->getFee();
    // $this->getTotalToReceive();
    // Recalculate when transfer data changes
    if (in_array($field, ['amount', 'receiver_fee'])) {
      //    unset($this->totalToReceive);
    }
    //  $this->calclulateDabitandCredit();
  }
  public function updatedTransferReceiverId($value)
  {
    if ($value) {
      [$source, $agentId] = explode('_', $value);
      $this->source = $source ?? "internal";
      $this->agentId = $agentId;
      if ($source == "internal") {
        $receiver = Agent::with('address')->find($agentId);
        $this->receiver_currencies = $receiver->getCanReceiveCurrency();

        $this->receiverDetails = [
          'name' => $receiver->name,
          'phone' => $receiver->phone,
          'address' => $receiver->address?->full_address,
        ];
        $this->receiverAgent = $receiver;
        $this->address = $receiver->getFullAddress();
      } else {
        $config = ApiConfig::find($source);
        $api = ApiConnection($config);
        $data = $api->getClientData($agentId);
        $this->address = $data['address'] ?? "";
      }
    }
  }



  // When currency changes
  public function updatedTransferCurrency()
  {
    $this->transfer['delivery_currency'] = $this->transfer['currency'];
  }

  public function getTotalToReceive()
  {
    $amount = $this->transfer['amount'] ?? 0;
    $receiverFee = $this->transfer['receiver_fee'] ?? 0;
    //$total = max(0, $amount - $receiverFee);
    $total = $amount;
    $this->totalToReceive = $total;
  }

  // Calculate total amount to be deducted from sender
  public function getTotalToSendProperty()
  {
    $amount = $this->transfer['amount'] ?? 0;
    $senderFee = $this->transfer['sender_fee'] ?? 0;
    return $amount + $senderFee;
  }

  // Reset sender details
  protected function resetSenderDetails()
  {
    $this->senderDetails = [
      'name' => null,
      'phone' => null,
      'address' => null,
      'balance' => null,
    ];
  }

  // Reset receiver details
  protected function resetReceiverDetails()
  {
    $this->receiverDetails = [
      'name' => null,
      'phone' => null,
      'address' => null,
    ];
  }
  public function calclulateDabitandCredit()
  {
    $amount = $this->transfer['amount'] ?? 0;
    $receiverFee = $this->transfer['receiver_fee'] ?? 0;
    $senderFee = $this->transfer['sender_fee'] ?? 0;
    $debit = $amount + $senderFee;
    $credit = $amount + $receiverFee;
    $revenue = $senderFee - $receiverFee;
    $this->debit = $debit;
    $this->credit = $credit;
    $this->revenue = $revenue;
  }
  public  function  SendTransfer()
  {
    if (!isset($this->transfer['client_id'])) {
      $this->dispatchError(__("alerts.client_id_required"));
      return;
    }
    try {
      $transfer = $this->transfer;
      $transfer['secret'] = rand(1000, 9999);
      $transfer['reference'] = $this->generateRandomTransferNumber();

      $lock = Cache::lock('lock:transfer:' . $transfer['reference'], 3);
      if (!$lock->get()) {
        $this->dispatchError(__("alerts.transfer_locked"));
        return;
      }

      DB::beginTransaction();


      if ($this->source != "internal") {
        $result = $this->processExternalTransfer($transfer);
        if (!$result['success']) {
          DB::rollBack();
          $this->dispatchError($result['message']);
          return;
        }
        $transfer = array_merge($transfer, $result['data']);
      } else {
        $transfer['receiver_id'] = $this->receiverAgent->id;
      }
      if ($this->senderAgent->isitChild($this->receiverAgent->id)) {
        $this->dispatch("sweetalert:error", ['msg' => __("Cannot Transfer To Your Branch"), "title" => "Error", "type" => "error"]);
        return;
      }

      // Create transfer
      $transfer['customer_details'] = $this->recipient;
      $new_transfer = (new TransferService())->create($transfer);

      DB::commit();
      $this->dispatch('ShowTransferReceipt', ['transfer_id' => $new_transfer->id]);

      // Defer non-critical tasks
      $this->dispatch('saveMedia', \App\Models\LedgerTransfer::class, $new_transfer->id);
      $this->updateAddressIfChanged($new_transfer, $this->address);
      $this->dispatch('RefreshDashboardStatistics');
      event(new RefreshDataTable('transfer-table'));
      event(new RefreshDataTable('collecttransfer-table'));
      event(new RefreshDataTable('livewatch-table'));
      // Show receipt immediately

      $this->resetForm();
    } catch (ValidationException $e) {
      $this->handleValidationError($e);
    } catch (CreditLimitExceededException $e) {
      DB::rollBack();
      $this->dispatchError($e->getMessage());
    } catch (\Exception $e) {
      DB::rollBack();
      Log::error("Transfer failed: " . $e->getMessage());
      $this->dispatchError($e->getMessage());
    } finally {
      optional($lock ?? null)->release();
    }

    $this->dispatch("UnBlockUI");
  }
  public function confirmTransfer()
  {
    if (!isset($this->transfer['client_id'])) {
      $this->dispatchError(__("alerts.client_id_required"));
      return;
    }

    $transfer = $this->transfer;
    $transfer['secret'] = rand(1000, 9999);
    $transfer['reference'] = $this->generateRandomTransferNumber();

    $rules = [
      'transfer.sender_id' => 'required|exists:agents,id',
      'transfer.receiver_id' => 'required|different:transfer.sender_id',
      'transfer.currency' => 'required|exists:ledger_currencies,code',
      'transfer.amount' => 'required|numeric|min:1',
      'transfer.send_fee' => 'required|numeric|min:0',
      'transfer.receiver_fee' => 'required|numeric|min:0',
      'transfer.recipient_name' => 'required',
      'transfer.recipient_phone' => 'required',
    ];

    try {
      $this->validate($rules);

      // التحقق من اسم المستفيد باستخدام النظام الجديد
      $recipientName = $this->transfer['recipient_name'] ?? '';
      $beneficiaryResult = TransferValidationHelper::validateBeneficiaryName($recipientName);
      if (!$beneficiaryResult['valid']) {
        $errorMessage = implode(', ', $beneficiaryResult['errors']) ?: 'اسم المستفيد غير صحيح';
        $this->dispatchError($errorMessage);
        return;
      }

      // التحقق من رقم الهاتف باستخدام النظام الجديد
      $recipientPhone = $this->transfer['recipient_phone'] ?? '';
      $phoneResult = TransferValidationHelper::validatePhoneNumber($recipientPhone);
      if (!$phoneResult['valid']) {
        $errorMessage = implode(', ', $phoneResult['errors']) ?: 'رقم الهاتف غير صحيح';
        $this->dispatchError($errorMessage);
        return;
      }
      if (!$this->senderAgent->canSendIn($this->transfer['currency'])) {
        $this->dispatchError(__("alerts.sender_currency_not_allowed"));
        return;
      }

      if ($this->source == "internal" && !$this->receiverAgent->canReceiveIn($transfer['delivery_currency'])) {
        $this->dispatchError(__("alerts.receiver_currency_not_allowed"));
        return;
      }
      (new TransferCreditLimitValidator())->validate(
        $this->senderAgent,
        $transfer['currency'],
        $transfer['amount'] + ($transfer['send_fee'] ?? 0)
      );

      $currency_word = $this->getAmountInWords($this->transfer['amount'], $this->transfer['currency']);
      $recipientName = $this->transfer['recipient_name'] ?? 'غير محدد';
      $this->dispatch("areyousure", ['msg' => __("alerts.areyousure"), 'emitName' => 'SendTransfer', 'action' => __("YES"), 'title' => __("alerts.areyousure_send_transfer", ["currency_word" => $currency_word, 'name' => $recipientName]), 'data' => []]);
    } catch (ValidationException $e) {
      $this->handleValidationError($e);
    } catch (CreditLimitExceededException $e) {
      DB::rollBack();
      $this->dispatchError($e->getMessage());
    } catch (\Exception $e) {
      DB::rollBack();
      Log::error("Transfer failed: " . $e->getMessage());
      $this->dispatchError($e->getMessage());
    } finally {
      optional($lock ?? null)->release();
    }
  }

  // Helper methods
  private function processExternalTransfer(array $transfer): array
  {
    $config = ApiConfig::with("apiType")->find((int)$this->source);
    $transfer['receiver_id'] = $config->agent_id;
    $result = sendExternalTransaction($config, $transfer, $this->agentId);

    return [
      'success' => $result['success'],
      'message' => $result['message'] ?? 'External transfer failed',
      'data' => [
        'api_id' => $config->id,
        'external_agent_id' => $this->agentId,
        'external_ref' => $result['ref'],
        'secret' => $result['password'] ?: rand(1000, 9999),
        'receiver_fee' => $result['tax'] ?? 0,
      ],
    ];
  }

  private function updateAddressIfChanged($transfer, $newAddress)
  {
    if (!empty($newAddress)) {
      $transfer->createOrUpdateAddress(['full_address' => $newAddress]);
    }
  }

  private function dispatchError(string $message)
  {
    $this->dispatch("sweetalert:error", [
      'msg' => $message,
      'title' => "Error",
      'type' => "error"
    ]);
  }

  private function handleValidationError(ValidationException $e)
  {
    $firstField = array_key_first($e->validator->errors()->messages());
    $this->dispatch('focus-error', ['field' => $firstField]);

    $firstErrorMessage = $e->validator->errors()->first($firstField);
    $this->dispatch("sweetalert:error", [
      'msg' => $firstErrorMessage,
      'title' => __("alerts.error"),
      'type' => "error"
    ]);
  }
  public  function UpdateTransfer()
  {
    try {
      LedgerTransfer::enableAuditing();
      $transfer = LedgerTransfer::find($this->transfer['id']);
      $this->transfer['receiver_id'] = $this->receiverAgent->id;
      $transfer->update($this->transfer);

      if (isset($transfer['external_ref'])) {
        $receiver = $transfer->receiver;
        $config = ApiConfig::with("apiType")->where("agent_id", $receiver->id)->first();
        $class = $config->apiType->class_name;
        $service_api = new $class();
        $configArray = json_decode($config->config_json, true);
        $configArray['base_url'] = $config->base_url;
        $service_api->setConfig($configArray);
        $res = $service_api->UpdateTransaction($transfer);
      } else {
        $service = new TransferService();
        $res = $service->UpdateLedger($transfer);
      }
      if ($res['success']) {
        $this->dispatch("sweetalert:success", ['msg' => $res['message'], "title" => "Success", "type" => "success"]);
      } else {
        $this->dispatch("sweetalert:error", ['msg' => $res['message'], "title" => "Error", "type" => "error"]);
      }
    } catch (\Exception $e) {
      Log::info($e);
      Log::error('Error updating transfer', ['error' => $e->getMessage()]);
    }
  }

  public function  getFee()
  {

    $sender = Agent::find($this->transfer['sender_id']);
    if ($sender) {
      $package = $sender->CurrencyTransferPackages($this->transfer['currency']);

      if ($package && $this->transfer['amount'] != 0 && isset($this->transfer['amount'])) {
        $fees = getFeePriceFromPackage($package->id, $this->transfer['amount']);
        if (!$fees['status']) {

          $this->addError('fee', $fees['msg']);
          $this->send_flag = false;
          return;
        }
        $this->send_flag = true;

        $this->transfer['send_fee'] = $fees['send_fee'];
        $this->transfer['receiver_fee'] = $fees['receive_fee'];
      } elseif ($this->transfer['amount'] != 0 && isset($this->transfer['amount'])) {
        $this->addError('fee', "No transfer package found");

        $this->send_flag = false;
      }
    }
  }
  public function getCanSendProperty(): bool
  {
    return $this->send_flag;
  }
  public function CheckBalance()
  {
    try {
      $currency = $this->transfer['currency'] ?? "";
      $amount = $this->transfer['amount'];
      $send_fee = $this->transfer['send_fee'] ?? 0;
      $agent = $this->senderAgent;

      if ($agent) {
        $creditlimitValidator = new TransferCreditLimitValidator();
        $creditlimitValidator->validate($this->senderAgent, $currency, $amount + $send_fee);
        $this->send_flag = true;
        Log::debug("CheckBalance: No exception, send_flag set to true");
      } else {
        $this->addError('balance', "Agent not found");
        $this->send_flag = false;
        Log::debug("CheckBalance: Agent not found, send_flag set to false");
      }
    } catch (CreditLimitExceededException $e) {
      Log::debug("CheckBalance: Credit limit exception caught, setting send_flag to false");
      Log::error($e);
      $this->addError('balance', $e->getMessage());
      $this->send_flag = false;
    } catch (\Exception $e) {
      Log::debug("CheckBalance: General exception caught, setting send_flag to false");
      Log::error($e);
      $this->send_flag = false;
      $this->addError('balance', $e->getMessage());
    }

    Log::debug("CheckBalance END: send_flag is " . var_export($this->send_flag, true));
  }


  // Reset the form
  public function resetForm()
  {
    $this->reset('transfer');
    $this->resetSenderDetails();
    $this->resetReceiverDetails();
    $this->client = null;
    $this->recipient = [
      'name' => '',
      'phone' => '',
      'sender_identity_number' => '',
    ];
    $this->transfer['sender_id'] = auth()->guard('agent')->user()->agent->id ?? 1;

    $this->transfer['transfer_date'] = now()->format('Y-m-d\TH:i');
    $this->resetErrorBag();
    $this->dispatch("ResetClientInfo");
  }
  public function getAmountInWordsProperty()
  {
    if ($this->transfer['amount'] && $this->transfer['currency']) {
      return toCurrencyWords(
        floatval($this->transfer['amount']),
        $this->transfer['currency'],
        app()->getLocale()
      );
    }
    return '';
  }
  public function getAmountInWords($amount, $currency)
  {
    if ($amount && $currency) {
      return toCurrencyWords(
        floatval($amount),
        $currency,
        app()->getLocale()
      );
    }
    return '';
  }
  public function  getBalances($agent_id)
  {
    if (!$agent_id) {
      return [];
    }
    $agent = Agent::find($agent_id);
    return $agent->getAgentBalances();
  }

  public function fieldIsVisible($key)
  {
    return $this->formFieldConfigs[$key]['visible'] ?? true;
  }

  public function fieldIsRequired($key)
  {
    return $this->formFieldConfigs[$key]['required'] ?? false;
  }

  public function fieldIsLocked($key)
  {
    return $this->formFieldConfigs[$key]['always_required'] ?? false;
  }
  public  function getFieldLabel($key)
  {
    return $this->formFieldConfigs[$key]['label'] ?? Str::title($key);
  }
  public  function getFieldTabIndex($key)
  {
    return $this->formFieldConfigs[$key]['tab_index'] ?? 1;
  }
  public function UpdatedRecipientSenderIdentityNumber()
  {
    $this->searchSender();
  }
  public function generateRandomTransferNumber(): string
  {
    $branchCode = str_pad(mt_rand(0, 999), 3, '0', STR_PAD_LEFT);
    $transactionId = str_pad(mt_rand(0, 999999), 6, '0', STR_PAD_LEFT);

    return "TRF-{$branchCode}-{$transactionId}";
  }

  public function render()
  {
    return view('livewire.agent-panel.transfer.add');
  }
}
