<?php

declare(strict_types=1);

namespace App\Livewire\AgentPanel\Booking;

use App\Models\Agent;
use App\Models\Currency;
use App\Models\FormFieldConfig;
use App\Models\LedgerCashier;
use App\Services\Transfer\CashierLedgerService;
use Livewire\Component;
use Illuminate\Support\Str;
use Livewire\WithFileUploads;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\ValidationException;
use Illuminate\Contracts\View\View as ViewContract;

class Add extends Component
{
  use WithFileUploads;
  //#[Validate('mimes:jpeg,png,jpg,pdf|max:1024')] // 1MB Max
  public $filters = ['type' => 'booking'];
  public  $type = "booking";
  public $booking_id = null;
  public $totalToReceive = 0;
  public $send_fee_type = "fixed", $receiver_fee_type = "fixed";
  public $debit = 0, $credit = 0, $transferFee = 0, $total = 0, $revenue = 0;
  public $searchUrl = '/api/agents/search';
  public $transfer = [
    'sender_id' => null,
    'receiver_id' => null,
    'currency' => 'USD', // default currency
    'delivery_currency' => 'USD',
    'delivery_amount' => 0,
    'amount' => null,
    'send_fee' => 0,
    'receiver_fee' => 0,
    'date' => null,
    'note' => null,
    'type' => "booking",
    'exchange_rate' => 1,
    'send_fee_value' => 0,
    'receiver_fee_value' => 0,

  ];
  public  $recipient = [
    'name' => '',
    'phone' => '',
  ];
  // Auto-filled sender details
  public $senderDetails = [
    'name' => null,
    'phone' => null,
    'address' => null,
    'balance' => null,
  ];

  // Auto-filled receiver details
  public $receiverDetails = [
    'name' => null,
    'phone' => null,
    'address' => null,
  ];

  // Available currencies and agents
  public $currencies = [];
  public $senders = [];
  public $receivers = [];
  public $formFieldConfigs = [];
  public $factor = "multiply";
  protected $rules = [
    'transfer.sender_id' => 'required|exists:agents,id',
    'transfer.receiver_id' => 'required|exists:agents,id|different:transfer.sender_id',
    'transfer.currency' => 'required',
    'transfer.amount' => 'required|numeric|min:0.01',
    'transfer.send_fee' => 'required|numeric|min:0',
    'transfer.receiver_fee' => 'required|numeric|min:0',
    'transfer.date' => 'nullable|date',
    'recipient.name' => 'required',
    'recipient.phone' => 'nullable',
  ];
  public array $attatchments = [];

  protected function getListeners(): array
  {
    return [
      'editTransfer' => 'EditBooking',
    ];
  }
  public function EditBooking(int $booking_id): void
  {

    $transfer = LedgerCashier::find($booking_id);
    if (!$transfer) {
      return;
    }

    $this->senderDetails = [
      'name' => $transfer->sender?->name,
      'phone' => $transfer->sender?->phone,
      'address' => $transfer->sender?->address?->full_address,
      'balance' => 0, // $transfer->sender?->getBalance($transfer->currency),
    ];
    $this->receiverDetails = [
      'name' => $transfer->receiver?->name,
      'phone' => $transfer->receiver?->phone,
      'address' => $transfer->receiver?->address?->full_address,
    ];
    $this->recipient = [
      'name' => $transfer->customer_details['name'],
      'phone' => $transfer->customer_details['phone'],

    ];
    $this->transfer = $transfer->toArray();

    $this->transfer['receiver_fee_value'] = $transfer->receiver_fee;

    $this->transfer['send_fee_value'] = $transfer->send_fee;
    $this->send_fee_type = 'fixed';
    $this->receiver_fee_type = 'fixed';
    // dd($this->transfer);
    $this->dispatch("scrollToForm", ['form_id' => "booking_form"]);
  }
  public function  UploadAttatchments(int $transfer_id): void
  {
    $transfer = LedgerCashier::find($transfer_id);
    if (!$transfer) {
      return;
    }

    if (count($this->attatchments) > 0) {
      $transfer->clearMediaCollection('attatchments');
      foreach ($this->attatchments as $attatchment) {

        $transfer
          ->addMedia($attatchment->getRealPath())
          ->usingFileName($attatchment->getClientOriginalName())
          ->toMediaCollection('attatchments');
      }
    }
  }
  public function fieldIsVisible(string $key): bool
  {
    return (bool)($this->formFieldConfigs[$key]['visible'] ?? true);
  }

  public function fieldIsRequired(string $key): bool
  {
    return (bool)($this->formFieldConfigs[$key]['required'] ?? false);
  }

  public function fieldIsLocked(string $key): bool
  {
    return (bool)($this->formFieldConfigs[$key]['always_required'] ?? false);
  }
  public  function getFieldLabel(string $key): string
  {
    return (string)($this->formFieldConfigs[$key]['label'] ?? Str::title($key));
  }

  public  function UpdatedTransferNote(string $value): void
  {
    $this->transfer['receiver_note'] = $value;
  }
  public function mount(): void
  {

    $type = $this->filters['type'] ?? 'booking';
    $this->type = $type;
    $this->transfer['date'] = now()->format('Y-m-d');
    $this->currencies = Currency::get();
    $this->transfer['currency'] = $this->currencies[0]->code ?? "USD";


    $this->transfer['date'] = now()->format('Y-m-d');
    $this->formFieldConfigs = FormFieldConfig::forForm('booking_form');
    if ($this->booking_id) {
      $this->EditBooking($this->booking_id);
    }
  }
  public function Updated(string $key, $value): void
  {

    if (str_contains($key, 'receiver_id')) return;

    if ($key != "factor")
      $this->factor = getCurrencyFactor($this->transfer['currency'], $this->transfer['delivery_currency'], 'transfer');
    if ($this->transfer['currency'] == $this->transfer['delivery_currency']) {
      $this->transfer['exchange_rate'] = 1;
      $this->transfer['delivery_amount'] = $this->transfer['amount'];
    }
    $this->CalculateDeliveryAmount();
    $this->calculateFees();
    //  $this->validate();
  }
  public function UpdatedTransferAmount($value): void
  {
    $this->transfer['amount'] = $value === '' ? null : $value;

    $this->CalculateDeliveryAmount();
    $this->calculateFees();
  }


  public function updatedTransferSenderId($value): void
  {

    if ($value) {
      $sender = Agent::with('address')->find($value);
      $this->senderDetails = [
        'name' => $sender?->name,
        'phone' => $sender?->phone,
        'address' => $sender?->address?->full_address,
        'balance' => 0, // $sender->getBalance($this->transfer['currency']),
      ];
    } else {
      $this->resetSenderDetails();
    }
  }
  public function updatedTransfer($value, string $field): void
  {
    $this->getTotalToReceive();
    // Recalculate when transfer data changes
    if (in_array($field, ['amount', 'receiver_fee'])) {
      //    unset($this->totalToReceive);
    }
    //  $this->calclulateDabitandCredit();
  }
  public function updatedTransferReceiverId($value): void
  {
    if ($value) {
      $receiver = Agent::with('address')->find($value);
      $this->receiverDetails = [
        'name' => $receiver?->name,
        'phone' => $receiver?->phone,
        'address' => $receiver?->address?->full_address,
      ];
      $this->calculateFees();
    } else {
      $this->resetReceiverDetails();
    }
  }





  // When currency changes
  public function updatedTransferCurrency(): void
  {
    $this->transfer['delivery_currency'] = $this->transfer['currency'];
    $this->calculateFees();
    // Update sender balance display if sender is selected
    // if ($this->transfer['sender_id']) {
    //     $sender = Agent::find($this->transfer['sender_id']);
    //     $this->senderDetails['balance'] = $sender->getBalance($this->transfer['currency']);
    // }
    // $this->calclulateDabitandCredit();
  }


  public function getTotalToReceive(): void
  {
    $amount = $this->transfer['amount'] ?? 0;
    $receiverFee = $this->transfer['receiver_fee'] ?? 0;
    //$total = max(0, $amount - $receiverFee);
    $total = $amount;
    $this->totalToReceive = $total;
  }

  // Calculate total amount to be deducted from sender
  public function getTotalToSendProperty(): float
  {
    $amount = $this->transfer['amount'] ?? 0;
    $senderFee = $this->transfer['sender_fee'] ?? 0;
    return (float)$amount + (float)$senderFee;
  }

  // Reset sender details
  protected function resetSenderDetails(): void
  {
    $this->senderDetails = [
      'name' => null,
      'phone' => null,
      'address' => null,
      'balance' => null,
    ];
  }

  // Reset receiver details
  protected function resetReceiverDetails(): void
  {
    $this->receiverDetails = [
      'name' => null,
      'phone' => null,
      'address' => null,
    ];
  }
  public function calclulateDabitandCredit()
  {
    $amount = $this->transfer['amount'] ?? 0;
    $receiverFee = $this->transfer['receiver_fee'] ?? 0;
    $senderFee = $this->transfer['sender_fee'] ?? 0;
    $debit = $amount + $senderFee;
    $credit = $amount + $receiverFee;
    $revenue = $senderFee - $receiverFee;
    $this->debit = $debit;
    $this->credit = $credit;
    $this->revenue = $revenue;
  }
  public function confirmTransfer(): void
  {


    try {
      $this->validate();
      $this->transfer['reference'] = generateRandomTransferNumber("REC");
      $this->transfer['secret'] = rand(1000, 9999);
      $this->transfer['user_id'] = Auth::id() ?? 1;
      $transfer = $this->transfer;


      $transfer['customer_details'] = $this->recipient;

      LedgerCashier::disableAuditing();
      if (isset($transfer['id'])) {
        $this->UpdateBooking();
        return;
      } else {
        $ledgercashier = new LedgerCashier();
        $transfer = $ledgercashier->create($transfer);
        $service = new CashierLedgerService();
        $service->createCashierLedgerEntries($transfer, "booking");
      }

      $this->dispatch('saveMedia', \App\Models\LedgerBooking::class, $transfer->id);


      $this->dispatch("toastr:success", ['msg' =>  __("transfers.transfer_created_successfully_toast", ['reference' => $transfer->reference, 'amount' => $transfer->amount, 'currency' => $transfer->currency]), 'reference' => $transfer->reference, 'amount' => $transfer->amount, 'currency' => $transfer->currency, "title" =>  __("alerts.success"), "type" => "success"]);
      $this->resetForm();
    } catch (ValidationException $e) {
      $firstField = array_key_first($e->validator->errors()->messages());
      $firstErrorMessage = $e->validator->errors()->first($firstField);
      $this->dispatch('focus-error', ['field' => $firstField]);
      $this->dispatch("sweetalert:error", [
        'msg' => $firstErrorMessage,
        'title' => __("alerts.error"),
        'type' => "error"
      ]);
      throw $e; // rethrow so Livewire still shows errors
    } catch (\Exception $e) {
      Log::info($e);
      // session()->flash('error', 'Transfer failed: ' . $e->getMessage());
    }
  }
  public function cancel()
  {
    return redirect()->to('/Booking');
  }
  public  function UpdateBooking(): void
  {
    try {
      $transfer = LedgerCashier::find($this->transfer['id']);
      LedgerCashier::enableAuditing();
      $transfer->update($this->transfer);

      $service = new CashierLedgerService();
      $res = $service->createCashierLedgerEntries($transfer);
      $this->UploadAttatchments($transfer->id);
      if (is_array($res) && ($res['success'] ?? false)) {
        $this->dispatch("sweetalert:success", ['msg' => ($res['message'] ?? ''), "title" => __("Success"), "type" => "success"]);
        redirect()->to('/Booking');
      } else {
        Log::info('Cashier ledger response', ['res' => $res]);
        $this->dispatch("sweetalert:error", ['msg' => (is_array($res) ? ($res['message'] ?? __('alerts.something_went_wrong')) : __('alerts.something_went_wrong')), "title" => __("Error"), "type" => "error"]);
      }
    } catch (\Exception $e) {
      Log::info($e);
      Log::error('Error updating transfer', ['error' => $e->getMessage()]);
    }
  }
  // Reset the form
  public function resetForm(): void
  {
    $this->reset('transfer');
    $this->resetSenderDetails();
    $this->resetReceiverDetails();
    $this->transfer['currency'] = 'USD';
    $this->transfer['date'] = now()->format('Y-m-d');
    $this->recipient = [
      'name' => '',
      'phone' => '',
    ];
    $this->resetErrorBag();
  }
  public function getAmountInWordsProperty(): string
  {
    if ($this->transfer['amount'] && $this->transfer['currency']) {
      return toCurrencyWords(
        floatval($this->transfer['amount']),
        $this->transfer['currency']
      );
    }
    return '';
  }
  public function getAmountInWords($amount, string $currency): string
  {
    if ($amount && $currency) {
      return toCurrencyWords(
        floatval($amount),
        $currency
      );
    }
    return '';
  }
  public  function UpdatedSendFeeType(string $value): void
  {
    $this->send_fee_type = $value;
    $this->calculateFees();
  }
  public function UpdatedReceiveFeeType(string $value): void
  {
    $this->receiver_fee_type = $value;
    $this->calculateFees();
  }
  public function calculateFees(): void
  {
    $amount = (float)($this->transfer['amount'] ?? 0);
    $sendFeeValue = (float)($this->transfer['send_fee_value'] ?? 0);
    $receiverFeeValue = (float)($this->transfer['receiver_fee_value'] ?? 0);

    $this->transfer['send_fee'] = match ($this->send_fee_type) {
      'fixed' => $sendFeeValue,
      'percent_hundred' => $amount * $sendFeeValue / 100,
      'percent_thousand' => $amount * $sendFeeValue / 1000,
      'percent_ten_thousand' => $amount * $sendFeeValue / 10000,
      default => 0,
    };

    $this->transfer['receiver_fee'] = match ($this->receiver_fee_type) {
      'fixed' => $receiverFeeValue,
      'percent_hundred' => $amount * $receiverFeeValue / 100,
      'percent_thousand' => $amount * $receiverFeeValue / 1000,
      'percent_ten_thousand' => $amount * $receiverFeeValue / 10000,
      default => 0,
    };
  }
  public function parseFeeShortcut($value, string $key): ?array
  {
    if (preg_match('/^(\d+(?:\.\d+)?)([a-zA-Z])$/', trim($value), $matches)) {
      $amount = (float) $matches[1];
      $type = strtoupper($matches[2]);

      return match ($type) {
        'L' => [$this->$key = "percent_hundred"],
        'F' => [$this->$key => 'percent_thousand', 'value' => $amount],
        'A' => [$this->$key => 'percent_ten_thousand', 'value' => $amount],
        default => [$this->$key => 'fixed', 'value' => $amount],
      };
    }

    //return ['type' => 'fixed', 'value' => (float) $value]; // لا يوجد اختصار
    return null;
  }

  public  function  UpdatedTransferExchangeRate($value): void
  {

    $this->CalculateDeliveryAmount();
  }

  public function UpdatedFactor($value): void
  {
    $this->CalculateDeliveryAmount();
  }
  public  function CalculateDeliveryAmount(): void
  {
    $this->transfer['delivery_amount'] = Exchange($this->transfer['amount'] ?? 0, $this->transfer['exchange_rate'] ?? 1, $this->factor ?? "multiply");
  }
  public function RoundAmount(string $key): void
  {
    if (isset($this->transfer[$key])) {
      $this->transfer[$key] = round((float)$this->transfer[$key]);
    }
  }
  public function  getBalances($agent_id): array
  {
    if (!$agent_id) {
      return [];
    }
    $agent = Agent::find($agent_id);
    if (!$agent) {
      return [];
    }
    $balances = $agent->getAgentBalances();
    return is_array($balances) ? $balances : $balances->toArray();
  }

  public function render(): ViewContract
  {
    return view('livewire.agent-panel.booking.add');
  }
}
