<?php

namespace App\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Redis;
use App\Helpers\RedisSessionHelper;

class LogoutAgentUsersJob implements ShouldQueue
{
  use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

  protected $agentId;
  protected $agentName;

  /**
   * Create a new job instance.
   */
  public function __construct($agentId, $agentName = null)
  {
    $this->agentId = $agentId;
    $this->agentName = $agentName;
  }

  /**
   * Execute the job.
   */
  public function handle(): void
  {
    try {
      Log::info("Starting logout process for agent ID: {$this->agentId}");

      // Get all users associated with this agent
      $users = DB::table('users')
        ->where('agent_id', $this->agentId)
        ->whereNotNull('agent_id')
        ->get();

      $logoutCount = 0;

      foreach ($users as $user) {
        if ($this->logoutUser($user->id)) {
          $logoutCount++;
        }
      }

      Log::info("Successfully logged out {$logoutCount} users for agent ID: {$this->agentId}");
    } catch (\Exception $e) {
      Log::error("Error in LogoutAgentUsersJob for agent ID {$this->agentId}: " . $e->getMessage());
      throw $e;
    }
  }

  /**
   * Logout a specific user
   */
  private function logoutUser($userId): bool
  {
    try {
      // Invalidate personal access tokens
      $tokensDeleted = DB::table('personal_access_tokens')
        ->where('tokenable_type', 'App\Models\User')
        ->where('tokenable_id', $userId)
        ->delete();

      // Clear user's cache
      Cache::forget("user_session_{$userId}");
      Cache::forget("agent_session_{$userId}");

      // Clear Redis sessions and user data using helper
      RedisSessionHelper::clearUserSessions($userId);

      Log::info("User ID {$userId} logged out successfully. Tokens deleted: {$tokensDeleted}");

      return true;
    } catch (\Exception $e) {
      Log::error("Error logging out user ID {$userId}: " . $e->getMessage());
      return false;
    }
  }
}
