<?php

namespace App\Http\Controllers;

use App\DataTables\AgentDataTable;
use App\DataTables\BranchDataTable;
use App\DataTables\CashierDataTable;
use App\DataTables\CreditSettingDataTable;
use App\DataTables\DeletedAgentsDataTable;
use App\DataTables\SypRatesDataTable;
use App\Models\Agent;
use App\Models\ApiConfig;
use App\Models\LedgerApproval;
use App\Models\LedgerBooking;
use App\Models\LedgerSyp;
use App\Models\LedgerTransfer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class AgentController extends Controller
{
  public  function  index(Request $request)
  {
    $type = $request->type;
    $id = $request->id;
    $filters = [];
    if ($type == 'cashier') {
      $dataTable = new CashierDataTable();
    } else if ($type == 'Branches') {
      $dataTable = new BranchDataTable($filters);
    } else
      $dataTable = new AgentDataTable();
    if ($id) {
      $filters =  ['agent_id' => $id];
    }
    return $dataTable->render('Agents.index', compact('filters'));
  }
  public  function deleted(Request $request)
  {
    $title = __("Deleted Agents");



    $dataTable = new DeletedAgentsDataTable();

    return $dataTable->render('Agents.deleted', compact('title'));
  }
  public  function branches(Request $request)
  {
    $filters = [];
    $id = $request->id;

    if ($id) {
      $filters =  ['parent_id' => $id];
      $title = __("Branches");
    }
    $dataTable = new BranchDataTable($filters);
    return $dataTable->render('Agents.branches', compact('filters'));
  }
  public function CreditLimits(CreditSettingDataTable $dataTable)
  {
    return $dataTable->render('Agents.CreditLimits');
  }
  public function Search(Request $request)
  {



    $api_id = $request->api_id;
    $currency = $request->currency;

    $query = $request->search;
    $type = $request->type;
    $direction = $request->direction;
    if ($api_id) {

      $api_id = $request->api_id;

      $config = ApiConfig::find($api_id);

      $class = $config->apiType->class_name;



      $service = new $class();
      $configArray = json_decode($config->config_json, true);
      $configArray['base_url'] = $config->base_url;

      $service->setConfig($configArray);
      $service->authenticate();
      $clients = $service->getClientList($query, $currency);
      return response()->json(['results' => $clients]);
    }

    $results = $this->getAgentCollection($type, $direction, $query);

    return response()->json(['results' => $results]);
  }
  public function AgentSearch(Request $request)
  {
    $type = $request->type ?? "transfer";
    $direction = $request->direction ?? "sender";
    $search = $request->input('search');
    $page = $request->input('page', 1);
    $perPage = 10;

    // Handle request for specific ID
    if ($request->has('id')) {
      return $this->handleSingleAgentRequest($request);
    }

    // Get the appropriate agent collection
    $agents = $this->getAgentCollection($type, $direction);

    // Apply search filter if provided
    if ($search) {
      $agents = $agents->filter(function ($agent) use ($search) {
        return stripos($agent['text'], $search) !== false;
      });
    }

    // Manual pagination for collections
    $total = $agents->count();
    $paginatedItems = $agents->forPage($page, $perPage)->values();

    return response()->json([
      'results' => $paginatedItems,
      'pagination' => [
        'more' => ($page * $perPage) < $total
      ]
    ]);
  }
  public function AgentSypRates(SypRatesDataTable $dataTable)
  {
    return $dataTable->render('Agents.syprates');
  }



  protected function handleSingleAgentRequest(Request $request)
  {
    $id = $request->id;

    // Handle plain numeric IDs (e.g., "7")
    if (is_numeric($id)) {
      $agent = Agent::with("address")->find($id);
      return response()->json([
        'results' => $agent ? [[
          'id' => $agent->id, // Return plain ID to match what was sent
          'text' => $agent->name . " - " . $agent->code . " - " . $agent->address?->country ?? "" . "-" . $agent->address?->city ?? "",
          'status' => 'active',
          'source' => 'internal',
          'agent_id' => $agent->id
        ]] : [],
        'pagination' => ['more' => false]
      ]);
    }

    // Handle prefixed IDs (internal_X or external_X)
    $parts = explode('_', $id);

    // Internal agent (internal_7)
    if ($parts[0] === 'internal' && isset($parts[1])) {
      $agent = Agent::with("address")->find($parts[1]);
      return response()->json([
        'results' => $agent ? [[
          'id' => $id, // Return the exact same ID format that was requested
          'text' => $agent->name . " - " . $agent->code . " - " . $agent->address?->country ?? "" . "-" . $agent->address?->city ?? "",
          'status' => 'active',
          'source' => 'internal',
          'agent_id' => $agent->id
        ]] : [],
        'pagination' => ['more' => false]
      ]);
    }

    // External agent (external_5)
    if ($parts[0] === 'external' && isset($parts[1])) {
      $externalAgent = Agent::find($parts[1]);
      return response()->json([
        'results' => $externalAgent ? [[
          'id' => $id, // Return the exact same ID format
          'text' => $externalAgent->name,
          'status' => 'active',
          'source' => 'external',
          'agent_id' => $externalAgent->external_id
        ]] : [],
        'pagination' => ['more' => false]
      ]);
    }

    // Fallback for invalid formats
    return response()->json([
      'results' => [],
      'pagination' => ['more' => false]
    ]);
  }

  protected function getAgentCollection($type, $direction, $search = null)
  {

    Log::info('Getting agent collection', ['type' => $type, 'direction' => $direction]);
    if ($type === 'transfer') {
      if ($direction === 'sender') {
        return collect(LedgerTransfer::GetSourceAgents())
          ->map(function ($agent) {
            return [
              'id'       => $agent->id,
              'text'     => $agent->name . " - " . $agent->code . " - " . ($agent->address?->country ?? "") . "-" . ($agent->address?->city ?? ""),
              'status'   => 'active',
              'source'   => 'internal',
              'agent_id' => $agent->id,
            ];
          })
          ->filter(function ($agent) use ($search) {
            // normalize and make sure $search is not null
            if (empty($search)) {
              return true; // no search → keep all
            }
            return str_contains(strtolower($agent['text']), strtolower($search));
          })
          ->values(); // reset keys (0,1,2…)
      } else {
        return LedgerTransfer::GetDestinationAgents();
      }
    } elseif ($type === 'approval') {
      if ($direction === 'sender') {
        return collect(LedgerApproval::GetSourceAgents())
          ->map(function ($agent) {
            return [
              'id'       => $agent->id,
              'text'     => $agent->name . " - " . $agent->code . " - " . ($agent->address?->country ?? "") . "-" . ($agent->address?->city ?? ""),
              'status'   => 'active',
              'source'   => 'internal',
              'agent_id' => $agent->id,
            ];
          })
          ->filter(function ($agent) use ($search) {
            // normalize and make sure $search is not null
            if (empty($search)) {
              return true; // no search → keep all
            }
            return str_contains(strtolower($agent['text']), strtolower($search));
          })
          ->values(); // reset keys (0,1,2…)
      } else {

        return LedgerApproval::GetDestinationAgents()
          ->map(function ($agent) {
            return [
              'id'       => $agent->id,
              'text'     => $agent->name . " - " . $agent->code . " - " . ($agent->address?->country ?? "") . "-" . ($agent->address?->city ?? ""),
              'status'   => 'active',
              'source'   => 'internal',
              'agent_id' => $agent->id,
            ];
          })
          ->filter(function ($agent) use ($search) {
            // normalize and make sure $search is not null
            if (empty($search)) {
              return true; // no search → keep all
            }
            return str_contains(strtolower($agent['text']), strtolower($search));
          })
          ->values(); // reset keys (0,1,2…)
      }
    } elseif ($type === 'booking') {
      if ($direction === 'sender') {

        return collect(LedgerBooking::GetSourceAgents())->map(function ($agent) {
          return [
            'id' =>   $agent->id,
            'text' => $agent->name . " - " . $agent->code . " - " . $agent->address?->country ?? "" . "-" . $agent->address?->city ?? "",

            'status' => 'active',
            'source' => 'internal',
            'agent_id' => $agent->id
          ];
        });
      } else {
        return collect(LedgerBooking::GetDestinationAgents())->map(function ($agent) {
          return [
            'id' =>   $agent->id,
            'text' => $agent->name . " - " . $agent->code . " - " . $agent->address?->country ?? "" . "-" . $agent->address?->city ?? "",
            'status' => 'active',
            'source' => 'internal',
            'agent_id' => $agent->id
          ];
        });
      }
    } elseif ($type === 'payment') {
      if ($direction === 'sender') {
        return collect(LedgerBooking::GetSourceAgents())->map(function ($agent) {
          return [
            'id' =>   $agent->id,
            'text' => $agent->name . " - " . $agent->code . " - " . $agent->address?->country ?? "" . "-" . $agent->address?->city ?? "",

            'status' => 'active',
            'source' => 'internal',
            'agent_id' => $agent->id
          ];
        });
      } else {
        return collect(LedgerBooking::GetDestinationAgents("payment"))->map(function ($agent) {
          return [
            'id' =>   $agent->id,
            'text' => $agent->name . " - " . $agent->code . " - " . $agent->address?->country ?? "" . "-" . $agent->address?->city ?? "",
            'status' => 'active',
            'source' => 'internal',
            'agent_id' => $agent->id
          ];
        });
      }
    } elseif ($type === 'collection') {

      if ($direction === 'sender') {
        return collect(LedgerBooking::GetSourceAgents("collection"))->map(function ($agent) {
          return [
            'id' =>   $agent->id,
            'text' => $agent->name . " - " . $agent->code . " - " . $agent->address?->country ?? "" . "-" . $agent->address?->city ?? "",

            'status' => 'active',
            'source' => 'internal',
            'agent_id' => $agent->id
          ];
        });
      } else {
        return collect(LedgerBooking::GetDestinationAgents())->map(function ($agent) {
          return [
            'id' =>   $agent->id,
            'text' => $agent->name . " - " . $agent->code . " - " . $agent->address?->country ?? "" . "-" . $agent->address?->city ?? "",
            'status' => 'active',
            'source' => 'internal',
            'agent_id' => $agent->id
          ];
        });
      }
    } elseif ($type === 'payment') {

      if ($direction === 'sender') {
        return collect(LedgerBooking::GetSourceAgents())->map(function ($agent) {
          return [
            'id' =>   $agent->id,
            'text' => $agent->name . " - " . $agent->code . " - " . $agent->address?->country ?? "" . "-" . $agent->address?->city ?? "",

            'status' => 'active',
            'source' => 'internal',
            'agent_id' => $agent->id
          ];
        });
      } else {
        return collect(LedgerBooking::GetDestinationAgents("payment"))->map(function ($agent) {
          return [
            'id' =>   $agent->id,
            'text' => $agent->name . " - " . $agent->code . " - " . $agent->address?->country ?? "" . "-" . $agent->address?->city ?? "",
            'status' => 'active',
            'source' => 'internal',
            'agent_id' => $agent->id
          ];
        });
      }
    } elseif ($type === 'syp') {
      if ($direction === 'sender') {
        return collect(LedgerBooking::GetSourceAgents())->map(function ($agent) {
          return [
            'id' =>   $agent->id,
            'text' => $agent->name . " - " . $agent->code . " - " . $agent->address?->country ?? "" . "-" . $agent->address?->city ?? "",
            'status' => 'active',
            'source' => 'internal',
            'agent_id' => $agent->id
          ];
        });
      } else {
        return collect(LedgerSyp::GetDestinationAgents())->map(function ($agent) {
          return [
            'id' =>   $agent->id,
            'text' => $agent->name . " - " . $agent->code . " - " . $agent->address?->country ?? "" . "-" . $agent->address?->city ?? "",
            'status' => 'active',
            'source' => 'internal',
            'agent_id' => $agent->id
          ];
        });
      }
    }
    // Default case
    return collect(LedgerTransfer::GetSourceAgents())->map(function ($agent) {
      return [
        'id' =>  $agent->id,
        'text' => $agent->name . " - " . $agent->code . " - " . $agent->address?->country ?? "" . "-" . $agent->address?->country ?? "",
        'status' => 'active',
        'source' => 'internal',
        'agent_id' => $agent->id
      ];
    });
  }

  /**
   * Get agent balances for API endpoint
   */
  public function getBalances($id)
  {
    try {
      $agent = Agent::findOrFail($id);

      // Get balances using the Agent model method
      $balances = $agent->getAgentBalances();
      $limit = \getBalanceLimit($agent->id);
      $basebalance = \getBaseCurrencyBalance($balances);

      // Generate HTML for the balance display
      $html = view('components.balance-badges', compact('balances', 'limit', 'basebalance'))->render();

      return response()->json([
        'success' => true,
        'html' => $html
      ]);
    } catch (\Exception $e) {
      Log::error("Failed to get agent balances for ID {$id}: " . $e->getMessage());

      return response()->json([
        'success' => false,
        'message' => 'Error loading balances: ' . $e->getMessage()
      ], 500);
    }
  }
}
