<?php

namespace App\Helpers;

use Rawilk\Settings\Facades\Settings;

class TransferValidationHelper
{
  /**
   * Get beneficiary name validation rules
   */
  public static function getBeneficiaryNameRules(): array
  {
    return [
      'min_words' => Settings::get('transfer_validation.beneficiary_name_min_words', 2),
      'max_words' => Settings::get('transfer_validation.beneficiary_name_max_words', 10),
      'required' => Settings::get('transfer_validation.require_beneficiary_name', true),
      'allow_special_chars' => Settings::get('transfer_validation.allow_special_chars_in_name', false),
    ];
  }

  /**
   * Get phone number validation rules
   */
  public static function getPhoneNumberRules(): array
  {
    return [
      'min_chars' => Settings::get('transfer_validation.phone_number_min_chars', 8),
      'max_chars' => Settings::get('transfer_validation.phone_number_max_chars', 15),
      'required' => Settings::get('transfer_validation.require_phone_number', true),
      'format' => Settings::get('transfer_validation.phone_number_format', 'international'),
    ];
  }

  /**
   * Validate beneficiary name
   */
  public static function validateBeneficiaryName(string $name): array
  {
    $rules = self::getBeneficiaryNameRules();
    $words = explode(' ', trim($name));
    $wordCount = count(array_filter($words, 'strlen')); // Count non-empty words

    $errors = [];

    if ($rules['required'] && empty(trim($name))) {
      $errors[] = 'اسم المستفيد مطلوب';
    }

    if ($wordCount < $rules['min_words']) {
      $errors[] = "اسم المستفيد يجب أن يحتوي على {$rules['min_words']} كلمة على الأقل";
    }

    if ($wordCount > $rules['max_words']) {
      $errors[] = "اسم المستفيد يجب أن يحتوي على {$rules['max_words']} كلمة كحد أقصى";
    }

    if (!$rules['allow_special_chars'] && !preg_match('/^[\p{L}\s]+$/u', $name)) {
      $errors[] = 'اسم المستفيد يجب أن يحتوي على أحرف عربية أو إنجليزية فقط';
    }

    return [
      'valid' => empty($errors),
      'errors' => $errors,
      'word_count' => $wordCount,
      'min_required' => $rules['min_words'],
      'max_allowed' => $rules['max_words'],
    ];
  }

  /**
   * Validate phone number
   */
  public static function validatePhoneNumber(string $phone): array
  {
    $rules = self::getPhoneNumberRules();
    $phone = trim($phone);

    $errors = [];

    if ($rules['required'] && empty($phone)) {
      $errors[] = 'رقم الهاتف مطلوب';
    }

    if (!empty($phone)) {
      $length = strlen($phone);

      if ($length < $rules['min_chars']) {
        $errors[] = "رقم الهاتف يجب أن يحتوي على {$rules['min_chars']} حرف على الأقل";
      }

      if ($length > $rules['max_chars']) {
        $errors[] = "رقم الهاتف يجب أن يحتوي على {$rules['max_chars']} حرف كحد أقصى";
      }

      // Format validation based on settings
      if (!self::validatePhoneFormat($phone, $rules['format'])) {
        $errors[] = self::getPhoneFormatErrorMessage($rules['format']);
      }
    }

    return [
      'valid' => empty($errors),
      'errors' => $errors,
      'length' => strlen($phone),
      'min_required' => $rules['min_chars'],
      'max_allowed' => $rules['max_chars'],
      'format' => $rules['format'],
    ];
  }

  /**
   * Validate phone number format
   */
  private static function validatePhoneFormat(string $phone, string $format): bool
  {
    switch ($format) {
      case 'international':
        // International format: +966501234567
        return preg_match('/^\+\d{1,4}\d{6,14}$/', $phone);

      case 'local':
        // Local format: 0501234567 (Saudi Arabia)
        return preg_match('/^05\d{8}$/', $phone);

      case 'flexible':
        // Flexible: any format with digits
        return preg_match('/^[\d\s\-\+\(\)]+$/', $phone);

      default:
        return true;
    }
  }

  /**
   * Get phone format error message
   */
  private static function getPhoneFormatErrorMessage(string $format): string
  {
    switch ($format) {
      case 'international':
        return 'رقم الهاتف يجب أن يكون بالتنسيق الدولي (+966501234567)';

      case 'local':
        return 'رقم الهاتف يجب أن يكون بالتنسيق المحلي (0501234567)';

      case 'flexible':
        return 'رقم الهاتف يجب أن يحتوي على أرقام فقط';

      default:
        return 'تنسيق رقم الهاتف غير صحيح';
    }
  }

  /**
   * Get Laravel validation rules for forms
   */
  public static function getLaravelValidationRules(): array
  {
    $beneficiaryRules = self::getBeneficiaryNameRules();
    $phoneRules = self::getPhoneNumberRules();

    $rules = [];

    if ($beneficiaryRules['required']) {
      $rules['beneficiary_name'] = 'required|string';

      // Add custom validation rule for word count
      $rules['beneficiary_name'] .= "|min_words:{$beneficiaryRules['min_words']},{$beneficiaryRules['max_words']}";

      if (!$beneficiaryRules['allow_special_chars']) {
        $rules['beneficiary_name'] .= '|regex:/^[\p{L}\s]+$/u';
      }
    }

    if ($phoneRules['required']) {
      $rules['phone_number'] = 'required|string';
      $rules['phone_number'] .= "|min:{$phoneRules['min_chars']}|max:{$phoneRules['max_chars']}";

      // Add format validation
      if ($phoneRules['format'] === 'international') {
        $rules['phone_number'] .= '|regex:/^\+\d{1,4}\d{6,14}$/';
      } elseif ($phoneRules['format'] === 'local') {
        $rules['phone_number'] .= '|regex:/^05\d{8}$/';
      }
    }

    return $rules;
  }

  /**
   * Get validation messages for Laravel
   */
  public static function getLaravelValidationMessages(): array
  {
    return [
      'beneficiary_name.required' => 'اسم المستفيد مطلوب',
      'beneficiary_name.min_words' => 'اسم المستفيد يجب أن يحتوي على :min كلمة على الأقل و :max كلمة كحد أقصى',
      'beneficiary_name.regex' => 'اسم المستفيد يجب أن يحتوي على أحرف عربية أو إنجليزية فقط',

      'phone_number.required' => 'رقم الهاتف مطلوب',
      'phone_number.min' => 'رقم الهاتف يجب أن يحتوي على :min حرف على الأقل',
      'phone_number.max' => 'رقم الهاتف يجب أن يحتوي على :max حرف كحد أقصى',
      'phone_number.regex' => 'تنسيق رقم الهاتف غير صحيح',
    ];
  }
}
