<?php

namespace App\Exports;

use Maatwebsite\Excel\Concerns\FromArray;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\WithColumnWidths;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Concerns\WithMapping;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Fill;

class TrialBalanceExport implements FromArray, WithHeadings, WithStyles, WithColumnWidths, WithTitle, WithMapping
{
    protected $data;
    protected $currency;
    protected $fromDate;
    protected $toDate;

    public function __construct(array $data, string $currency = '', string $fromDate = '', string $toDate = '')
    {
        $this->data = $data;
        $this->currency = $currency;
        $this->fromDate = $fromDate;
        $this->toDate = $toDate;
    }

    public function array(): array
    {
        return $this->data;
    }

    public function headings(): array
    {
        return [
            'كود الحساب',
            'اسم الحساب',
            'نوع الحساب',
            'العمق',
            'رصيد مدين',
            'رصيد دائن',
            'الرصيد الصافي',
            'رصيد الافتتاح',
            'عدد الحركات',
            'معلومات إضافية'
        ];
    }

    public function map($row): array
    {
        return [
            $row['account_code'] ?? '',
            $row['account_name'] ?? '',
            $row['account_type'] ?? '',
            $row['depth'] ?? '',
            number_format($row['debit_balance'] ?? 0, 2),
            number_format($row['credit_balance'] ?? 0, 2),
            number_format($row['net_balance'] ?? 0, 2),
            number_format($row['opening_balance'] ?? 0, 2),
            $row['movements']['transaction_count'] ?? 0,
            $this->formatExtraInfo($row['extra_info'] ?? null)
        ];
    }

    public function styles(Worksheet $sheet)
    {
        // Header styling
        $sheet->getStyle('A1:J1')->applyFromArray([
            'font' => [
                'bold' => true,
                'color' => ['rgb' => 'FFFFFF'],
            ],
            'fill' => [
                'fillType' => Fill::FILL_SOLID,
                'startColor' => ['rgb' => '4472C4'],
            ],
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_CENTER,
                'vertical' => Alignment::VERTICAL_CENTER,
            ],
            'borders' => [
                'allBorders' => [
                    'borderStyle' => Border::BORDER_THIN,
                    'color' => ['rgb' => '000000'],
                ],
            ],
        ]);

        // Data styling
        $lastRow = count($this->data) + 1;
        if ($lastRow > 1) {
            $sheet->getStyle("A2:J{$lastRow}")->applyFromArray([
                'borders' => [
                    'allBorders' => [
                        'borderStyle' => Border::BORDER_THIN,
                        'color' => ['rgb' => 'CCCCCC'],
                    ],
                ],
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_RIGHT,
                    'vertical' => Alignment::VERTICAL_CENTER,
                ],
            ]);

            // Number columns alignment
            $sheet->getStyle("E2:H{$lastRow}")->getAlignment()->setHorizontal(Alignment::HORIZONTAL_RIGHT);

            // Account code alignment
            $sheet->getStyle("A2:A{$lastRow}")->getAlignment()->setHorizontal(Alignment::HORIZONTAL_LEFT);
        }

        // Add summary row
        $this->addSummaryRow($sheet, $lastRow + 1);
    }

    public function columnWidths(): array
    {
        return [
            'A' => 15, // Account Code
            'B' => 30, // Account Name
            'C' => 15, // Account Type
            'D' => 10, // Depth
            'E' => 15, // Debit Balance
            'F' => 15, // Credit Balance
            'G' => 15, // Net Balance
            'H' => 15, // Opening Balance
            'I' => 12, // Movements Count
            'J' => 25, // Extra Info
        ];
    }

    public function title(): string
    {
        return 'الميزانية العمومية';
    }

    private function formatExtraInfo($extraInfo): string
    {
        if (!$extraInfo) {
            return '';
        }

        if (is_array($extraInfo)) {
            return json_encode($extraInfo, JSON_UNESCAPED_UNICODE);
        }

        if (is_string($extraInfo)) {
            return $extraInfo;
        }

        return (string) $extraInfo;
    }

    private function addSummaryRow(Worksheet $sheet, int $row)
    {
        $totalDebit = 0;
        $totalCredit = 0;
        $totalNet = 0;

        foreach ($this->data as $rowData) {
            $totalDebit += $rowData['debit_balance'] ?? 0;
            $totalCredit += $rowData['credit_balance'] ?? 0;
            $totalNet += $rowData['net_balance'] ?? 0;
        }

        // Add summary row
        $sheet->setCellValue("A{$row}", 'المجموع');
        $sheet->setCellValue("E{$row}", number_format($totalDebit, 2));
        $sheet->setCellValue("F{$row}", number_format($totalCredit, 2));
        $sheet->setCellValue("G{$row}", number_format($totalNet, 2));

        // Style summary row
        $sheet->getStyle("A{$row}:J{$row}")->applyFromArray([
            'font' => [
                'bold' => true,
            ],
            'fill' => [
                'fillType' => Fill::FILL_SOLID,
                'startColor' => ['rgb' => 'E7E6E6'],
            ],
            'borders' => [
                'allBorders' => [
                    'borderStyle' => Border::BORDER_THIN,
                    'color' => ['rgb' => '000000'],
                ],
            ],
        ]);

        // Add report info
        $this->addReportInfo($sheet, $row + 2);
    }

    private function addReportInfo(Worksheet $sheet, int $row)
    {
        $sheet->setCellValue("A{$row}", 'معلومات التقرير:');
        $sheet->setCellValue("A" . ($row + 1), 'العملة: ' . ($this->currency ?: 'غير محدد'));
        $sheet->setCellValue("A" . ($row + 2), 'من تاريخ: ' . ($this->fromDate ?: 'غير محدد'));
        $sheet->setCellValue("A" . ($row + 3), 'إلى تاريخ: ' . ($this->toDate ?: 'غير محدد'));
        $sheet->setCellValue("A" . ($row + 4), 'عدد الحسابات: ' . count($this->data));
        $sheet->setCellValue("A" . ($row + 5), 'تاريخ الإنشاء: ' . now()->format('Y-m-d H:i:s'));

        // Style report info
        $sheet->getStyle("A{$row}:A" . ($row + 5))->applyFromArray([
            'font' => [
                'bold' => true,
                'color' => ['rgb' => '4472C4'],
            ],
        ]);
    }
}

