<?php

namespace App\DataTables;

use App\Models\ReportExportLog;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Services\DataTable;
use Illuminate\Support\Facades\Storage;

class ReportExportedLogDataTable extends DataTable
{
    /**
     * Build the DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     */
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        return (new EloquentDataTable($query))
            ->addColumn('action', function ($row) {
                if ($row->status === 'completed') {
                    $url = Storage::disk('public')->url($row->file_path);
                    return '<a href="' . $url . '" class="btn btn-sm btn-outline-primary" download>Download</a>';
                }
                return '-';
            })
            ->editColumn('status', function ($row) {
                return match ($row->status) {
                    'completed' => '<span class="badge bg-success">Completed</span>',
                    'failed' => '<span class="badge bg-danger">Failed</span>',
                    default => '<span class="badge bg-warning">Pending</span>',
                };
            })
            ->rawColumns(['action', 'status']) // Allow HTML
            ->setRowId('id');
    }

    /**
     * Get the query source of dataTable.
     */
    public function query(ReportExportLog $model): QueryBuilder
    {
        return $model->newQuery();
    }

    /**
     * Optional method if you want to use the html builder.
     */
    public function html(): HtmlBuilder
    {
        return $this->builder()
            ->parameters([
                'language' => [
                    'url' => url('/vendor/datatables/lang/' . app()->getLocale() . '.json'),
                ],
                'headerCallback' => 'function(thead, data, start, end, display) {
    $(thead).css("background-color", "#685dd8");
    $(thead).find("th").css({
        "color": "white",
        "font-weight": "bold"
    });
}',
            ])
            ->setTableId('report-export-log-table')
            ->columns($this->getColumns())
            ->minifiedAjax()
            ->orderBy(0, 'desc')

            ->buttons([]);
    }

    /**
     * Get the dataTable columns definition.
     */
    public function getColumns(): array
    {
        return [
            Column::computed('action')
                ->title('Action')
                ->exportable(false)
                ->printable(false)
                ->width(100)
                ->addClass('text-center'),

            Column::make('id'),
            Column::make('type')->title('Report Type'),
            Column::make('from_date')->title('From Date'),
            Column::make('to_date')->title('To Date'),
            Column::make('status')->title('Status'),
            Column::make('created_at')->title('Created At'),
        ];
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'ReportExportLog_' . date('YmdHis');
    }
}
