<?php

namespace App\DataTables;

use App\Models\CustomAudit;
use App\Models\LedgerApproval;
use App\Models\LedgerBooking;
use App\Models\LedgerExchange;
use App\Models\LedgerTransfer;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Services\DataTable;
use Carbon\Carbon;
use OwenIt\Auditing\Models\Audit;
use Illuminate\Support\Facades\Log;

class AuditDataTable extends DataTable
{
  public function dataTable(QueryBuilder $query): EloquentDataTable
  {
    return (new EloquentDataTable($query))
      ->addColumn('user_info', function ($audit) {

        $user = $audit->user;
        if ($user) {
          return '<div class="d-flex align-items-center">
                                    <div class="avatar avatar-sm me-2">
                                        <span class="avatar-initial rounded-circle bg-label-primary">' . substr($user->name, 0, 1) . '</span>
                                    </div>
                                    <div>
                                        <span class="fw-semibold">' . $user->name . '</span><br>
                                        <small class="text-muted">' . $user->email . '</small>
                                    </div>
                                </div>';
        }

        return '<span class="text-muted">' . __('audit.unknown_user') . '</span>';
      })
      ->addColumn('agent_info', function ($audit) {

        $agent = null;
        if (isset($audit->auditable->sender) || isset($audit->auditable->receiver)) {

          $agent = $audit->auditable->sender ?? $audit->auditable->receiver;
        } elseif (isset($audit->auditable->user)) {
          $agent = $audit->auditable->user->agent;
        }
        if ($agent) {
          return '<span class="text-muted">' . $agent->name . '</span>';
        }
        return '<span class="text-muted">--</span>';
      })
      ->addColumn('event_badge', function ($audit) {
        $eventColors = [
          'created' => 'success',
          'updated' => 'warning',
          'deleted' => 'danger',
          'restored' => 'info'
        ];
        $eventColor = $eventColors[$audit->event] ?? 'secondary';
        return '<span class="badge bg-' . $eventColor . '">' . __('audit.events.' . $audit->event) . '</span>';
      })
      ->addColumn('model_badge', function ($audit) {
        $models = [
          'App\Models\LedgerTransfer' => __('audit.models.ledger_transfer'),
          'App\Models\LedgerBooking' => __('audit.models.ledger_booking'),
          'App\Models\Client' => __('audit.models.client'),
          'App\Models\User' => __('audit.models.user'),
          'App\Models\Agent' => __('audit.models.agent'),
          'App\Models\LedgerExchange' => __('audit.models.ledger_exchange'),
          'App\Models\LedgerApproval' => __('audit.models.ledger_approval'),
        ];
        $modelName = $models[$audit->auditable_type] ?? class_basename($audit->auditable_type);
        return '<span class="badge bg-label-info">' . $modelName . '</span>';
      })
      ->editColumn('browser ', function ($audit) {
        if ($audit->browser) {
          return '<div class="d-flex align-items-center">
                                <img src="/assets/img/icons/audit/' . ucfirst($audit->browser) . '.png"
                                     alt="' . $audit->browser . '"
                                     class="me-2"
                                     width="20"
                                     height="20">
                                <span>' . $audit->browser . '</span>
                            </div>';
        }
        return '<span class="text-muted">' . __('audit.unknown_browser') . '</span>';
      })
      ->addColumn('actions', function ($audit) {
        return view('Audits.actions', compact('audit'));
      })
      ->addColumn('transfer_description', function ($audit) {
        if (!$audit->auditable) {
          return '<span class="text-muted">--</span>';
        }

        switch (get_class($audit->auditable)) {
          case \App\Models\LedgerTransfer::class:
            $sender = $audit->auditable->sender?->name ?? 'N/A';
            $receiver = $audit->auditable->receiver?->name ?? 'N/A';
            return "{$audit->auditable->getDescription('out')} from <b>{$sender}</b> to <b>{$receiver}</b>";

          case \App\Models\LedgerBooking::class:
            return "Booking by <b>{$audit->auditable->user?->name}</b>";

          case \App\Models\LedgerApproval::class:
            return "Approval by <b>{$audit->auditable->user?->name}</b>";

          case \App\Models\LedgerExchange::class:
            return "Exchange handled by <b>{$audit->auditable->user?->name}</b>";

          default:
            return '<span class="text-muted">--</span>';
        }
      })
      ->addColumn('formatted_timestamp', function ($audit) {
        return '<div class="d-flex flex-column">
                            <span class="fw-semibold">' . $audit->created_at->format('H:i:s') . '</span>
                            <small class="text-muted">' . $audit->created_at->format('Y-m-d') . '</small>
                        </div>';
      })
      ->rawColumns(['user_info', 'event_badge', 'model_badge', 'browser', 'actions', 'formatted_timestamp', 'agent_info', 'transfer_description'])
      ->setRowId('id')
      ->addIndexColumn();
  }

  public function query(CustomAudit $query): QueryBuilder
  {
    $query = CustomAudit::select([
      'id',
      'user_type',
      'user_id',
      'event',
      'auditable_type',
      'auditable_id',
      'ip_address',
      'browser',
      'platform',
      'device_type',
      'country',
      'created_at'
    ])
      ->with([
        'user:id,name,email',
        'agent:id,name',

        'auditable' => function ($morphTo) {
          $morphTo->morphWith([
            \App\Models\LedgerTransfer::class => ['sender:id,name', 'receiver:id,name'],
            \App\Models\LedgerBooking::class => ['user.agent:id,name'],
            \App\Models\LedgerApproval::class => ['user.agent:id,name'],
            \App\Models\LedgerExchange::class => ['user.agent:id,name'],
          ]);
        },
      ]);
    // Apply filters from request



    $filters = [];
    if (request()->filled('filters')) {


      $filters = request()->get('filters');
    }

    $query = $query->Filter($filters);


    return $query;
  }

  public function html(): HtmlBuilder
  {
    return $this->builder()
      ->setTableId('audit-table')
      ->columns($this->getColumns())
      ->minifiedAjax()
      ->dom('<"row"' .
        '<"col-md-2"<"ms-n2"l>>' .
        '<"col-md-10"<"dt-action-buttons text-xl-end text-lg-start text-md-end text-start d-flex align-items-center justify-content-end flex-md-row flex-column mb-6 mb-md-0 mt-n6 mt-md-0"fB>>' .
        '>r' . // <-- add this line
        't' .
        '<"row"' .
        '<"col-sm-12 col-md-6"i>' .
        '<"col-sm-12 col-md-6"p>' .
        '>')
      ->orderBy(1, 'desc')
      ->ajax([
        'url' => '', // current URL
        'type' => 'GET',
        'data' => 'function(data) {
                 const urlParams = new URLSearchParams(window.location.search);
    const show = urlParams.get("show");
        data.category_ids = window.selectedCategoryIds || [];
        data.subcategory_ids = window.selectedSubcategoryIds || [];
        data.donation_types = window.selectedDonationTypes || [];
        data.filters = window.filters || {};
        if (show) {
        data.filters.show = show;
    };
        data.search_term = $("#general-search").val(); // input field for general search

    }'
      ])

      ->parameters([
        'scrollX' => true,
        'responsive' => true,
        'autoWidth' => false,
        'pageLength' => 25,
        'lengthMenu' => [[10, 25, 50, 100], [10, 25, 50, 100]],

      ]);
  }

  protected function getColumns(): array
  {
    return [
      Column::make('formatted_timestamp')
        ->title(__('audit.table.timestamp'))
        ->searchable(false)
        ->orderable(false),
      Column::make('user_info')
        ->title(__('audit.table.user'))
        ->searchable(false)
        ->orderable(false),
      Column::make('agent_info')
        ->title(__('audit.table.agent'))
        ->searchable(false)
        ->orderable(false),
      Column::make('event_badge')
        ->title(__('audit.table.event'))
        ->searchable(false)
        ->orderable(false),
      Column::make('model_badge')
        ->title(__('audit.table.model'))
        ->searchable(false)
        ->orderable(false),
      Column::make('ip_address')
        ->title(__('audit.table.ip_address')),
      Column::make('browser')
        ->title(__('audit.table.browser'))
        ->searchable(false)
        ->orderable(false),
      Column::make('transfer_description')
        ->title(__('audit.table.transfer_description'))
        ->searchable(false)
        ->orderable(false),
      Column::make('actions')
        ->title(__('audit.table.actions'))
        ->searchable(false)
        ->orderable(false)
        ->width(120),
    ];
  }

  protected function filename(): string
  {
    return 'AuditLogs_' . date('Y_m_d_His');
  }
}
