<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Laravel\Scout\EngineManager;
use Laravel\Scout\Searchable;

class ScoutSync extends Command
{
  /**
   * The name and signature of the console command.
   *
   * @var string
   */
  protected $signature = 'scout:sync-model {model? : Class name of model to update settings}';

  /**
   * The console command description.
   *
   * @var string
   */
  protected $description = 'Sync your configuration with Meilisearch';

  /**
   * The Meilisearch client.
   *
   * @var \Meilisearch\Client
   */
  private $client;

  public function __construct()
  {
    parent::__construct();
    $this->client = app(EngineManager::class)->driver('meilisearch');
  }

  /**
   * Execute the console command.
   */
  public function handle(): void
  {
    if ($namespace = $this->argument('model')) {
      $model = new $namespace;
      $this->syncModel($model);
      return;
    }

    $this->syncAll();
  }

  private function syncModel($model): void
  {
    if ($this->hasSettings($model)) {
      $this->updateSettings($model);
    }
  }

  private function syncAll(): void
  {
    collect(scandir(app_path('Models')))->each(function ($path) {
      if ($path === '.' || $path === '..') {
        return true;
      }

      $namespace = 'App\Models\\' . substr($path, 0, -4);
      $model = new $namespace;
      $this->syncModel($model);
    });
  }

  private function updateSettings($model): void
  {
    $index = $this->client->index($model->searchableAs());

    // Update settings
    $index->updateSettings($model->meilisearchSettings);

    // Confirm success
    $this->line("Settings updated for {$model->searchableAs()}");
  }

  private function hasSettings($model): bool
  {
    return in_array(Searchable::class, class_uses($model)) && property_exists($model, 'meilisearchSettings');
  }
}
